#ifndef ZLIBH
#define ZLIBH

#include "zconf.h"

#ifdef __cplusplus
extern "C" {
#endif

#define ZLIB_VERSION "1.2.5"
#define ZLIB_VERNUM 0x1250
#define ZLIB_VER_MAJOR 1
#define ZLIB_VER_MINOR 2
#define ZLIB_VER_REVISION 5
#define ZLIB_VER_SUBREVISION 0

typedef voidpf (*alloc_func) OF((voidpf opaque, uInt items, uInt size));
typedef void   (*free_func)  OF((voidpf opaque, voidpf address));

struct internal_state;

typedef struct z_stream_s {
    Bytef    *next_in;  /* next input byte */
    uInt     avail_in;  /* number of bytes available at next_in */
    uLong    total_in;  /* total nb of input bytes read so far */

    Bytef    *next_out; /* next output byte should be put there */
    uInt     avail_out; /* remaining free space at next_out */
    uLong    total_out; /* total nb of bytes output so far */

    char     *msg;      /* last error message, NULL if no error */
    struct internal_state FAR *state; /* not visible by applications */

    alloc_func zalloc;  /* used to allocate the internal state */
    free_func  zfree;   /* used to free the internal state */
    voidpf     opaque;  /* private data object passed to zalloc and zfree */

    int     data_type;  /* best guess about the data type: binary or text */
    uLong   adler;      /* adler32 value of the uncompressed data */
    uLong   reserved;   /* reserved for future use */
} z_stream;

typedef z_stream FAR *z_streamp;

/*
     gzip header information passed to and from zlib routines.  See RFC 1952
  for more details on the meanings of these fields.
*/
typedef struct gz_header_s {
    int     text;       /* true if compressed data believed to be text */
    uLong   time;       /* modification time */
    int     xflags;     /* extra flags (not used when writing a gzip file) */
    int     os;         /* operating system */
    Bytef   *extra;     /* pointer to extra field or Z_NULL if none */
    uInt    extra_len;  /* extra field length (valid if extra != Z_NULL) */
    uInt    extra_max;  /* space at extra (only when reading header) */
    Bytef   *name;      /* pointer to zero-terminated file name or Z_NULL */
    uInt    name_max;   /* space at name (only when reading header) */
    Bytef   *comment;   /* pointer to zero-terminated comment or Z_NULL */
    uInt    comm_max;   /* space at comment (only when reading header) */
    int     hcrc;       /* true if there was or will be a header crc */
    int     done;       /* true when done reading gzip header (not used
                           when writing a gzip file) */
} gz_header;

typedef gz_header FAR *gz_headerp;

/*
     The application must update next_in and avail_in when avail_in has dropped
   to zero.  It must update next_out and avail_out when avail_out has dropped
   to zero.  The application must initialize zalloc, zfree and opaque before
   calling the init function.  All other fields are set by the compression
   library and must not be updated by the application.

     The opaque value provided by the application will be passed as the first
   parameter for calls of zalloc and zfree.  This can be useful for custom
   memory management.  The compression library attaches no meaning to the
   opaque value.

     zalloc must return Z_NULL if there is not enough memory for the object.
   If zlib is used in a multi-threaded application, zalloc and zfree must be
   thread safe.

     On 16-bit systems, the functions zalloc and zfree must be able to allocate
   exactly 65536 bytes, but will not be required to allocate more than this if
   the symbol MAXSEG_64K is defined (see zconf.h).  WARNING: On MSDOS, pointers
   returned by zalloc for objects of exactly 65536 bytes *must* have their
   offset normalized to zero.  The default allocation function provided by this
   library ensures this (see zutil.c).  To reduce memory requirements and avoid
   any allocation of 64K objects, at the expense of compression ratio, compile
   the library with -DMAX_WBITS=14 (see zconf.h).

     The fields total_in and total_out can be used for statistics or progress
   reports.  After compression, total_in holds the total size of the
   uncompressed data and may be saved for use in the decompressor (particularly
   if the decompressor wants to decompress everything in a single step).
*/

												/* constants */

#define Z_NO_FLUSH      0
#define Z_PARTIAL_FLUSH 1
#define Z_SYNC_FLUSH    2
#define Z_FULL_FLUSH    3
#define Z_FINISH        4
#define Z_BLOCK         5
#define Z_TREES         6

#define Z_OK            0
#define Z_STREAM_END    1
#define Z_NEED_DICT     2
#define Z_ERRNO        (-1)
#define Z_STREAM_ERROR (-2)
#define Z_DATA_ERROR   (-3)
#define Z_MEM_ERROR    (-4)
#define Z_BUF_ERROR    (-5)
#define Z_VERSION_ERROR (-6)
/* Return codes for the compression/decompression functions. Negative values
 * are errors, positive values are used for special but normal events.
 */

#define Z_NO_COMPRESSION         0
#define Z_BEST_SPEED             1
#define Z_BEST_COMPRESSION       9
#define Z_DEFAULT_COMPRESSION  (-1)
/* compression levels */

#define Z_FILTERED            1
#define Z_HUFFMAN_ONLY        2
#define Z_RLE                 3
#define Z_FIXED               4
#define Z_DEFAULT_STRATEGY    0
/* compression strategy; see deflateInit2() below for details */

#define Z_BINARY   0
#define Z_TEXT     1
#define Z_ASCII    Z_TEXT   /* for compatibility with 1.2.2 and earlier */
#define Z_UNKNOWN  2
/* Possible values of the data_type field (though see inflate()) */

#define Z_DEFLATED   8
/* The deflate compression method (the only one supported in this version) */

#define Z_NULL  0  /* for initializing zalloc, zfree, opaque */

#define zlib_version zlibVersion()
/* for compatibility with versions < 1.0.2 */


												/* basic functions */


typedef const char * ZEXPORT (*dllzlibVersion) OF((void));

typedef int ZEXPORT (*dlldeflate) OF((z_streamp strm, int flush));

typedef int ZEXPORT (*dlldeflateEnd) OF((z_streamp strm));

typedef int ZEXPORT (*dllinflate) OF((z_streamp strm, int flush));

typedef int ZEXPORT (*dllinflateEnd) OF((z_streamp strm));

typedef int ZEXPORT (*dlldeflateSetDictionary) OF((z_streamp strm,
																						 const Bytef *dictionary,
																						 uInt  dictLength));

typedef int ZEXPORT (*dlldeflateCopy) OF((z_streamp dest,
																		z_streamp source));

typedef int ZEXPORT (*dlldeflateReset) OF((z_streamp strm));

typedef int ZEXPORT (*dlldeflateParams) OF((z_streamp strm,
																			int level,
																			int strategy));

typedef int ZEXPORT (*dlldeflateTune) OF((z_streamp strm,
																		int good_length,
																		int max_lazy,
																		int nice_length,
																		int max_chain));

typedef uLong ZEXPORT (*dlldeflateBound) OF((z_streamp strm,
																			 uLong sourceLen));

typedef int ZEXPORT (*dlldeflatePrime) OF((z_streamp strm,
																		 int bits,
																		 int value));

typedef int ZEXPORT (*dlldeflateSetHeader) OF((z_streamp strm,
																				 gz_headerp head));

typedef int ZEXPORT (*dllinflateSetDictionary) OF((z_streamp strm,
																						 const Bytef *dictionary,
																						 uInt  dictLength));

typedef int ZEXPORT (*dllinflateSync) OF((z_streamp strm));

typedef int ZEXPORT (*dllinflateCopy) OF((z_streamp dest,
																		z_streamp source));

typedef int ZEXPORT (*dllinflateReset) OF((z_streamp strm));

typedef int ZEXPORT (*dllinflateReset2) OF((z_streamp strm,
																			int windowBits));

typedef int ZEXPORT (*dllinflatePrime) OF((z_streamp strm,
																		 int bits,
																		 int value));

typedef long ZEXPORT (*dllinflateMark) OF((z_streamp strm));

typedef int ZEXPORT (*dllinflateGetHeader) OF((z_streamp strm,
																				 gz_headerp head));

typedef unsigned (*in_func) OF((void FAR *, unsigned char FAR * FAR *));
typedef int (*out_func) OF((void FAR *, unsigned char FAR *, unsigned));

typedef int ZEXPORT (*dllinflateBack) OF((z_streamp strm,
																		in_func in, void FAR *in_desc,
																		out_func out, void FAR *out_desc));

typedef int ZEXPORT (*dllinflateBackEnd) OF((z_streamp strm));

typedef uLong ZEXPORT (*dllzlibCompileFlags) OF((void));

typedef int ZEXPORT (*dllcompress) OF((Bytef *dest,   uLongf *destLen,
																 const Bytef *source, uLong sourceLen));

typedef int ZEXPORT (*dllcompress2) OF((Bytef *dest,   uLongf *destLen,
																	const Bytef *source, uLong sourceLen,
																	int level));

typedef uLong ZEXPORT (*dllcompressBound) OF((uLong sourceLen));

typedef int ZEXPORT (*dlluncompress) OF((Bytef *dest,   uLongf *destLen,
																	 const Bytef *source, uLong sourceLen));

typedef voidp gzFile;       /* opaque gzip file descriptor */


typedef gzFile ZEXPORT (*dllgzdopen) OF((int fd, const char *mode));

typedef int ZEXPORT (*dllgzbuffer) OF((gzFile file, unsigned size));

typedef int ZEXPORT (*dllgzsetparams) OF((gzFile file, int level, int strategy));

typedef int ZEXPORT (*dllgzread) OF((gzFile file, voidp buf, unsigned len));

typedef int ZEXPORT (*dllgzwrite) OF((gzFile file,
																voidpc buf, unsigned len));

typedef int ZEXPORTVA (*dllgzprintf) OF((gzFile file, const char *format, ...));

typedef int ZEXPORT (*dllgzputs) OF((gzFile file, const char *s));

typedef char * ZEXPORT (*dllgzgets) OF((gzFile file, char *buf, int len));

typedef int ZEXPORT (*dllgzputc) OF((gzFile file, int c));

typedef int ZEXPORT (*dllgzgetc) OF((gzFile file));

typedef int ZEXPORT (*dllgzungetc) OF((int c, gzFile file));

typedef int ZEXPORT (*dllgzflush) OF((gzFile file, int flush));

typedef int ZEXPORT    (*dllgzrewind) OF((gzFile file));

typedef int ZEXPORT (*dllgzeof) OF((gzFile file));

typedef int ZEXPORT (*dllgzdirect) OF((gzFile file));

typedef int ZEXPORT    (*dllgzclose) OF((gzFile file));

typedef int ZEXPORT (*dllgzclose_r) OF((gzFile file));
typedef int ZEXPORT (*dllgzclose_w) OF((gzFile file));

typedef const char * ZEXPORT (*dllgzerror) OF((gzFile file, int *errnum));

typedef void ZEXPORT (*dllgzclearerr) OF((gzFile file));

typedef uLong ZEXPORT (*dlladler32) OF((uLong adler, const Bytef *buf, uInt len));

typedef uLong ZEXPORT (*dllcrc32)   OF((uLong crc, const Bytef *buf, uInt len));

typedef int ZEXPORT (*dlldeflateInit_) OF((z_streamp strm, int level,
																		 const char *version, int stream_size));
typedef int ZEXPORT (*dllinflateInit_) OF((z_streamp strm,
																		 const char *version, int stream_size));
typedef int ZEXPORT (*dlldeflateInit2_) OF((z_streamp strm, int  level, int  method,
																			int windowBits, int memLevel,
																			int strategy, const char *version,
																			int stream_size));
typedef int ZEXPORT (*dllinflateInit2_) OF((z_streamp strm, int  windowBits,
																			const char *version, int stream_size));
typedef int ZEXPORT (*dllinflateBackInit_) OF((z_streamp strm, int windowBits,
																				 unsigned char FAR *window,
																				 const char *version,
																				 int stream_size));


#define deflateInit(strm, level) \
				deflateInit_((strm), (level),       ZLIB_VERSION, sizeof(z_stream))
#define inflateInit(strm) \
				inflateInit_((strm),                ZLIB_VERSION, sizeof(z_stream))
#define deflateInit2(strm, level, method, windowBits, memLevel, strategy) \
				deflateInit2_((strm),(level),(method),(windowBits),(memLevel),\
											(strategy),           ZLIB_VERSION, sizeof(z_stream))
#define inflateInit2(strm, windowBits) \
				inflateInit2_((strm), (windowBits), ZLIB_VERSION, sizeof(z_stream))
#define inflateBackInit(strm, windowBits, window) \
				inflateBackInit_((strm), (windowBits), (window), \
																						ZLIB_VERSION, sizeof(z_stream))

/*
#if defined(_LARGEFILE64_SOURCE) && _LFS64_LARGEFILE-0
	 ZEXTERN gzFile ZEXPORT gzopen64 OF((const char *, const char *));
	 ZEXTERN z_off64_t ZEXPORT gzseek64 OF((gzFile, z_off64_t, int));
	 ZEXTERN z_off64_t ZEXPORT gztell64 OF((gzFile));
	 ZEXTERN z_off64_t ZEXPORT gzoffset64 OF((gzFile));
	 ZEXTERN uLong ZEXPORT adler32_combine64 OF((uLong, uLong, z_off64_t));
	 ZEXTERN uLong ZEXPORT crc32_combine64 OF((uLong, uLong, z_off64_t));
#endif

#if !defined(ZLIB_INTERNAL) && _FILE_OFFSET_BITS-0 == 64 && _LFS64_LARGEFILE-0
#  define gzopen gzopen64
#  define gzseek gzseek64
#  define gztell gztell64
#  define gzoffset gzoffset64
#  define adler32_combine adler32_combine64
#  define crc32_combine crc32_combine64
#  ifdef _LARGEFILE64_SOURCE
		 ZEXTERN gzFile ZEXPORT gzopen64 OF((const char *, const char *));
		 ZEXTERN z_off_t ZEXPORT gzseek64 OF((gzFile, z_off_t, int));
		 ZEXTERN z_off_t ZEXPORT gztell64 OF((gzFile));
		 ZEXTERN z_off_t ZEXPORT gzoffset64 OF((gzFile));
		 ZEXTERN uLong ZEXPORT adler32_combine64 OF((uLong, uLong, z_off_t));
		 ZEXTERN uLong ZEXPORT crc32_combine64 OF((uLong, uLong, z_off_t));
#  endif
#else
	 ZEXTERN gzFile ZEXPORT gzopen OF((const char *, const char *));
	 ZEXTERN z_off_t ZEXPORT gzseek OF((gzFile, z_off_t, int));
	 ZEXTERN z_off_t ZEXPORT gztell OF((gzFile));
	 ZEXTERN z_off_t ZEXPORT gzoffset OF((gzFile));
	 ZEXTERN uLong ZEXPORT adler32_combine OF((uLong, uLong, z_off_t));
	 ZEXTERN uLong ZEXPORT crc32_combine OF((uLong, uLong, z_off_t));
#endif
*/

#if !defined(ZUTIL_H) && !defined(NO_DUMMY_DECL)
		struct internal_state {int dummy;};
#endif

/* undocumented functions */
typedef const char   * ZEXPORT (*dllzError)           OF((int));
typedef int            ZEXPORT (*dllinflateSyncPoint) OF((z_streamp));
typedef const uLongf * ZEXPORT (*dllget_crc_table)    OF((void));
typedef int            ZEXPORT (*dllinflateUndermine) OF((z_streamp, int));



int ZlibLoadLibrary(void);

int ZlibFreeLibrary(void);

int ZlibEnabled(void);

extern dllzlibVersion   				zlibVersion;
extern dlldeflate       				deflate;
extern dlldeflateEnd           deflateEnd;
extern dllinflate              inflate;
extern dllinflateEnd           inflateEnd;
extern dlldeflateSetDictionary deflateSetDictionary;
extern dlldeflateCopy          deflateCopy;
extern dlldeflateReset         deflateReset;
extern dlldeflateParams        deflateParams;
extern dlldeflateTune          deflateTune;
extern dlldeflateBound         deflateBound;
extern dlldeflatePrime         deflatePrime;
extern dlldeflateSetHeader     deflateSetHeader;
extern dllinflateSetDictionary inflateSetDictionary;
extern dllinflateSync          inflateSync;
extern dllinflateCopy          inflateCopy;
extern dllinflateReset         inflateReset;
extern dllinflateReset2        inflateReset2;
extern dllinflatePrime         inflatePrime;
extern dllinflateMark         	inflateMark;
extern dllinflateGetHeader     inflateGetHeader;
extern dllinflateBack          inflateBack;
extern dllinflateBackEnd       inflateBackEnd;
extern dllzlibCompileFlags     zlibCompileFlags;
extern dllcompress             compress;
extern dllcompress2            compress2;
extern dllcompressBound       	compressBound;
extern dlluncompress           uncompress;
extern dllgzdopen              gzdopen;
extern dllgzbuffer             gzbuffer;
extern dllgzsetparams          gzsetparams;
extern dllgzread               gzread;
extern dllgzwrite              gzwrite;
extern dllgzprintf             gzprintf;
extern dllgzputs               gzputs;
extern dllgzgets               gzgets;
extern dllgzputc               gzputc;
extern dllgzgetc               gzgetc;
extern dllgzungetc             gzungetc;
extern dllgzflush              gzflush;
extern dllgzrewind             gzrewind;
extern dllgzeof                gzeof;
extern dllgzdirect             gzdirect;
extern dllgzclose              gzclose;
extern dllgzclose_r            gzclose_r;
extern dllgzclose_w            gzclose_w;
extern dllgzerror              gzerror;
extern dllgzclearerr           gzclearerr;
extern dlladler32              adler32;
extern dllcrc32                crc32;
extern dlldeflateInit_         deflateInit_;
extern dllinflateInit_         inflateInit_;
extern dlldeflateInit2_        deflateInit2_;
extern dllinflateInit2_        inflateInit2_;
extern dllinflateBackInit_    	inflateBackInit_;
extern dllzError               zError;
extern dllinflateSyncPoint     inflateSyncPoint;
extern dllget_crc_table        get_crc_table;
extern dllinflateUndermine     inflateUndermine;


///////////////////////////////////////


#ifdef __cplusplus
}
#endif


#endif /* ZLIB_H */
