//-----------------------------------------------------------------------------
// Maya XFile Exporter
//-----------------------------------------------------------------------------

// TODO:
//  - texture placement
//  - blend shapes
//  - NURBS & SubD tesselation loses skinning & material info
//	- support for single-sided vs. double sided faces
//	- option to export only selected items




//-----------------------------------------------------------------------------
// INCLUDE FILES
//-----------------------------------------------------------------------------

#include <windows.h>
#include <iostream.h>
#include <math.h>

#include <initguid.h>
#include <rmxfguid.h>
#include <rmxftmpl.h>

// for dll export
//  2005/02/02 - saika
#include "defs.h"

#include "xskinexptemplates.h"
#include "xportTranslator.h"

// Dt API (included only for the DtMatrixGetTransforms() call)
#include "MDt.h"

// Maya API
#include <maya/MSimple.h>
#include <maya/MObject.h>
#include <maya/MIkSystem.h>
#include <maya/MDagPath.h>
#include <maya/MPlug.h>
#include <maya/MSelectionList.h>
#include <maya/MColor.h>
#include <maya/MMatrix.h>
#include <maya/MQuaternion.h>
#include <maya/MEulerRotation.h>
#include <maya/MTime.h>

#include <maya/MIntArray.h>
#include <maya/MFloatArray.h>
#include <maya/MFloatVectorArray.h>
#include <maya/MFloatPointArray.h>
#include <maya/MPointArray.h>
#include <maya/MObjectArray.h>
#include <maya/MDagPathArray.h>
#include <maya/MPlugArray.h>

#include <maya/MAnimControl.h>
#include <maya/MAnimUtil.h>

#include <maya/MFnPlugin.h>
#include <maya/MFnDagNode.h>
#include <maya/MFnTransform.h>
#include <maya/MFnIKJoint.h>
#include <maya/MFnMesh.h>
#include <maya/MFnNurbsSurface.h>
#include <maya/MFnSubd.h>
#include <maya/MFnLambertShader.h>
#include <maya/MFnBlinnShader.h>
#include <maya/MFnPhongShader.h>
#include <maya/MFnSkinCluster.h>
#include <maya/MFnWeightGeometryFilter.h>
#include <maya/MFnBlendShapeDeformer.h>
#include <maya/MFnMatrixData.h>
#include <maya/MFnSet.h>

#include <maya/MItSelectionList.h>
#include <maya/MItDependencyGraph.h>
#include <maya/MItDependencyNodes.h>
#include <maya/MItDag.h>
#include <maya/MItGeometry.h>
#include <maya/MItMeshPolygon.h>
#include <maya/MItMeshVertex.h>



//-----------------------------------------------------------------------------
// GLOBAL VARIABLES
//-----------------------------------------------------------------------------

CArrayTable g_Arrays;		// keeps some arrays (e.g. shape names) persistent
MDagPathArray g_AddedPaths;	// stores DAG paths of all exported shapes
DXFILEFORMAT g_FileFormat;	// ASCII, binary, or compressed file format
BOOL g_bExportSkin;			// export skinning
BOOL g_bExportPose;			// export pose information
BOOL g_bExportMaterials;	// export materials (and hence UVs)
BOOL g_bExportAnimation;	// export animations
BOOL g_bExportNURBS;		// export NURBS surfaces
BOOL g_bExportSubd;			// export subdivision surfaces
DWORD g_dwNURBSTess;		// 0 - untesselated, 1 - uniform, 2 - patch mesh
DWORD g_dwSubdTess;			// 0 - untesselated, 1 - adaptive, 2 - uniform
BOOL g_bKeyframeAnimation;	// TRUE - key framed, FALSE - fixed step animation
BOOL g_bAnimateEverything;	// useful for driven (indirect) animations
BOOL g_bRelativeTexFile;	// TRUE - relative, FALSE - absolute file names
DWORD g_iFrameStep;			// size of frame step for fixed step animation
FLOAT g_fFlipU, g_fFlipV;	// flip the U & V texture coordinate
const bool g_bIntermediateObjects_c = true;	// export intermediate objects
const GUID* g_aIds[] = { &DXFILEOBJ_XSkinMeshHeader,
					 	 &DXFILEOBJ_VertexDuplicationIndices,
						 &DXFILEOBJ_SkinWeights };	// array of templates



//-----------------------------------------------------------------------------
// HELPER FUNCTION PROTOTYPES
//-----------------------------------------------------------------------------

HRESULT AddScene
		(
			LPDIRECTXFILESAVEOBJECT pxofSave
		);
HRESULT	AddAnim
		(
			SAnim*					pAnim, 
			LPDIRECTXFILEDATA		pAnimSetObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		);
HRESULT AddTransform
		(
			MDagPath&				mdpTransform, 
			LPDIRECTXFILEDATA		pFrameDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		);
HRESULT AddShape
		(
			MDagPath&				mdpTransform,
			bool					bParentVisibility,
			LPDIRECTXFILEDATA		pParentFrameDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		);
HRESULT AddSubD
		(
			SShape*					pShape, 
			LPDIRECTXFILEDATA		pFrameDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		);
HRESULT AddNURBS
		(
			SShape*					pShape, 
			LPDIRECTXFILEDATA		pFrameDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		) ;
HRESULT AddPatches
		(
			SShape*					pShape, 
			LPDIRECTXFILEDATA		pFrameDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		) ;
HRESULT AddMesh
		(
			SShape*					pShape, 
			LPDIRECTXFILEDATA		pFrameDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		); 
HRESULT	AddNormals
		(
			SShape*					pShape, 
			LPDIRECTXFILEDATA		pShapeDataObject, 
			LPDIRECTXFILESAVEOBJECT pxofSave
		);
HRESULT AddUVs
		(
			SShape*					pShape, 
			LPDIRECTXFILEDATA		pShapeDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		);
HRESULT	AddMaterialList
		(
			SShape*					pShape, 
			LPDIRECTXFILEDATA		pShapeDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		);
HRESULT AddRepInfo
		(
			SShape*					pShape, 
			LPDIRECTXFILEDATA		pShapeDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		);
HRESULT	AddVertexColors
		(
			SShape*					pShape, 
			LPDIRECTXFILEDATA		pShapeDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		);
HRESULT AddSkin
		(
			SShape*					pShape, 
			LPDIRECTXFILEDATA		pShapeDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		);
HRESULT LoadTransformKey
		(
			MDagPath&	mdpTransform, 
			SKey*		pKey
		);
HRESULT LoadFixedStepAnims
		(
			DWORD*	pnAnims, 
			SAnim** paAnims
		);
HRESULT LoadKeyframedAnims
		(
			DWORD*	pnAnims, 
			SAnim** paAnims
		);
HRESULT LoadMesh
		(
			MDagPath&	mdpTransform, 
			MDagPath&	mdpMesh,
			SShape*		pShape
		);
HRESULT LoadSubd
		(
			MDagPath&	mdpTransform, 
			MDagPath&	mdpSubd,
			SShape*		pShape
		);
HRESULT LoadNURBS
		(
			MDagPath&	mdpTransform, 
			MDagPath&	mdpNurbs,
			SShape*		pShape
		);

int MyMatrixGetTransforms( float *pmatrix , float *translate , float *scale , float *quaternion , float *rotation )
{
	float matrix[ 4 ][ 4 ];
	for ( int i = 0 ; i < 16 ; ++i )
	{
		*( &matrix[ 0 ][ 0 ] + i ) = *( &pmatrix[ 0 ] + i );
	}
	
	MVector t;
	double d_scale[ 3 ];
	double d_quat[ 4 ];
	double euler[ 3 ];
	MStatus result;
	
	MMatrix mmatrix( matrix );
	
	MTransformationMatrix mtmatrix( mmatrix );
	if ( mtmatrix.getScale( d_scale , MSpace::Space::kTransform) != MS::kSuccess )
	{
		return 0;
	}
	
	t = mtmatrix.translation( MSpace::Space::kTransform , &result );
	if ( result != MS::kSuccess )
	{
		return 0;
	}
	
	if ( mtmatrix.getRotationQuaternion( d_quat[ 0 ] , d_quat[ 1 ] , d_quat[ 2 ] , d_quat[ 3 ] , MSpace::Space::kTransform ) != MS::kSuccess )
	{
		return 0;
	}
	
	MTransformationMatrix::RotationOrder rotation_order = MTransformationMatrix::RotationOrder::kLast;
	if ( mtmatrix.getRotation( euler , rotation_order ) != MS::kSuccess )
	{
		return 0;
	}
	
	translate[ 0 ] = static_cast< float >( t.x );
	translate[ 1 ] = static_cast< float >( t.y );
	translate[ 2 ] = static_cast< float >( t.z );
	scale[ 0 ] = static_cast< float >( d_scale[ 0 ] );
	scale[ 1 ] = static_cast< float >( d_scale[ 1 ] );
	scale[ 2 ] = static_cast< float >( d_scale[ 2 ] );
	quaternion[ 0 ] = static_cast< float >( d_quat[ 3 ] );
	quaternion[ 1 ] = static_cast< float >( d_quat[ 0 ] );
	quaternion[ 2 ] = static_cast< float >( d_quat[ 1 ] );
	quaternion[ 3 ] = static_cast< float >( d_quat[ 2 ] );
	rotation[ 0 ] = static_cast< float >( euler[ 0 ] );
	rotation[ 1 ] = static_cast< float >( euler[ 1 ] );
	rotation[ 2 ] = static_cast< float >( euler[ 2 ] );
	
	return 1;
}

//-----------------------------------------------------------------------------
// MAYA PLUGIN FUNCTIONS
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Name: initializePlugin()
// Desc: Initializes the Maya XFile Exporter plugin
//-----------------------------------------------------------------------------

DLL_EXPORT  MStatus initializePlugin(MObject obj)
{
	MStatus stat = MS::kSuccess;
	MFnPlugin plugin(obj, "XFileExporter", "4.0", "Any");

	// register the translator
	stat = plugin.registerFileTranslator("XFile",					// name
										 "xfileTranslator.rgb",		// icon
										 xfileTranslator::creator,	
										 "xfileTranslatorOpts",		// script
										 "",
										 true);
	if (!stat)
		stat.perror("registerFileTranslator");

	return stat;
}



//-----------------------------------------------------------------------------
// Name: uninitializePlugin()
// Desc: Uninitializes the Maya XFile Exporter plugin
//-----------------------------------------------------------------------------

DLL_EXPORT  MStatus	uninitializePlugin(MObject obj) 
{
	MStatus stat = MS::kSuccess;
	MFnPlugin plugin(obj);

	stat = plugin.deregisterFileTranslator("XFile");
	if (!stat)
		stat.perror("deregisterFileTranslator");

	return stat;
}



//-----------------------------------------------------------------------------
// XFILETRANSLATOR MEMBER FUNCTIONS
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Name: xfileTranslator::writer()
// Desc: Exports a Maya scene to an X file
//-----------------------------------------------------------------------------

MStatus	xfileTranslator::writer
		(
			const MFileObject& mfoXFile,			// save file object
			const MString& msOptions,				// options string
			MPxFileTranslator::FileAccessMode mode	// access mode (ignored)
		) 
{
	HRESULT hr = S_OK;
	MStatus stat = MS::kSuccess;
	MString	msFile, msExt;
    LPDIRECTXFILE pxofApi = NULL;
    LPDIRECTXFILESAVEOBJECT pxofSave = NULL; 
	MItDag itDag(MItDag::kDepthFirst, MFn::kTransform);
	MDagPathArray VisiblePaths;	// DAG paths of visible nodes
	DWORD i;

	cout << "\n\nMaya XFile Exporter\n";
	cout << "Copyright (C) 1998-2000 Microsoft Corporation. All Rights Reserved.\n\n";

	msFile = mfoXFile.fullName();
	msExt = msFile.substring(msFile.rindex('.'), msFile.length() - 1);

	if (msExt != ".x" && msExt != ".X") 
		msFile += ".x";

	g_Arrays.Init();

	// parse the options from the MEL UI script
	ParseOptions(msOptions);

	g_AddedPaths.clear();

	// create xofapi object.
	hr = DirectXFileCreate(&pxofApi);
	if (FAILED(hr))
		goto e_Exit;

	// register templates for d3drm.
	hr = pxofApi->RegisterTemplates((LPVOID)D3DRM_XTEMPLATES, D3DRM_XTEMPLATE_BYTES);
	if (FAILED(hr))
		goto e_Exit;

	// register extra templates for skinning info and vertex duplication
	hr = pxofApi->RegisterTemplates((LPVOID)XSKINEXP_TEMPLATES, strlen(XSKINEXP_TEMPLATES));
	if (FAILED(hr))
		goto e_Exit;

	// create save object.
	hr = pxofApi->CreateSaveObject(msFile.asChar(), g_FileFormat, &pxofSave);
	if (FAILED(hr))
		goto e_Exit;

	// save templates
	hr = pxofSave->SaveTemplates(3, g_aIds);
	if (FAILED(hr))
		goto e_Exit;
	
	cout << "Exporting to \"" << msFile.asChar() << "\"..." << endl;

	// hide all visible objects (this speeds up the animation export)
	for (itDag.reset(); !itDag.isDone(); itDag.next())
	{
		MPlug Visibility;
		MDagPath dagPath;
		MFnDagNode fnDag;
		bool bIsVisible;

		stat = itDag.getPath(dagPath);
		if (!stat)
			continue;
		
		stat = fnDag.setObject(dagPath);
		if (!stat)
			continue;

		Visibility = fnDag.findPlug("visibility", &stat);
		if (!stat)
			continue;

		stat = Visibility.getValue(bIsVisible);
		if (!stat || !bIsVisible)
			continue;

		stat = Visibility.setValue(false);
		if (!stat)
			continue;

		VisiblePaths.append(dagPath);
	}

	// export the scene 
	hr = AddScene(pxofSave);
	if (FAILED(hr))
		goto e_Exit;

e_Exit:
	// display previously visible objects
	for (i = 0; i < VisiblePaths.length(); i++)
	{
		MPlug Visibility;
		MFnDagNode fnDag;
		
		stat = fnDag.setObject(VisiblePaths[i]);
		if (!stat)
			continue;

		Visibility = fnDag.findPlug("visibility", &stat);
		if (!stat)
			continue;

		stat = Visibility.setValue(true);
		if (!stat)
			continue;
	}
	
	if (NULL != pxofSave) 
		pxofSave->Release();

	if (NULL != pxofApi)
		pxofApi->Release();

	g_Arrays.Nuke();	// delete persistent arrays

	if (FAILED(hr))
		cout << "\nThere were errors.\n";
	else 
		cout << "\n\"" << msFile.asChar() << "\" saved.\n\nCompleted successfully.\n";
	cout << "------------------------------------------------------------------------------------------------------\n\n";

	cout.flush();

	return MS::kSuccess;	// don't pass errors to Maya
}



//-----------------------------------------------------------------------------
// Name: xfileTranslator::ParseOptions()
// Desc: Parses a string of options and sets the global export variables
//-----------------------------------------------------------------------------

HRESULT xfileTranslator::ParseOptions(MString msOptions) 
{
	HRESULT hr = S_OK;
	MStringArray masOptionList;
	DWORD i;

	// set default options
	g_FileFormat = DXFILEFORMAT_TEXT;
	g_bExportSubd  = TRUE;
	g_bExportNURBS = TRUE;
	g_bExportSkin  = TRUE;
	g_bExportPose  = FALSE;
	g_bExportMaterials = TRUE;
	g_bExportAnimation = TRUE;
	g_dwSubdTess  = 1;		// adaptive tesselation
	g_dwNURBSTess = 1;		// uniform tesselation
	g_bKeyframeAnimation = TRUE;
	g_bAnimateEverything = TRUE;
	g_iFrameStep = 1;
	g_fFlipU = -1.0f;					
	g_fFlipV = 1.0f;
	g_bRelativeTexFile = FALSE;

	if (0 == msOptions.length()) 
		goto e_Exit;

	// split up all the options.
	msOptions.split(';', masOptionList);

	// parse each option
	for (i = 0; i < (int)masOptionList.length(); i++) 
	{
		MStringArray masOption;

		masOptionList[i].split('=', masOption);

		if (1 >= masOption.length()) 
			continue;	// ignore invalid option format

		if (masOption[0] == "fileFormat") 
		{
			if (masOption[1] == "binary") 
				g_FileFormat = DXFILEFORMAT_BINARY;
			else if (masOption[1] == "compressed") 
				g_FileFormat = DXFILEFORMAT_COMPRESSED;
		}
		else if (masOption[0] == "NURBSTess") 
		{
			if (masOption[1] == "none") 
				g_bExportNURBS = FALSE;
			else if (masOption[1] == "untesselated")
				g_dwNURBSTess = 0;
			else if (masOption[1] == "patch")
				g_dwNURBSTess = 2;
		}
		else if (masOption[0] == "SubDTess") 
		{
			if (masOption[1] == "none") 
				g_bExportSubd = FALSE;
			else if (masOption[1] == "untesselated")
				g_dwSubdTess = 0;
			else if (masOption[1] == "uniform")
				g_dwSubdTess = 2;
		}
		else if (masOption[0] == "exportSkin" && masOption[1] == "false") 
		{
				g_bExportSkin = FALSE;
		}
		else if (masOption[0] == "exportPose" && masOption[1] == "true") 
		{
				g_bExportPose = TRUE;
		}
		else if (masOption[0] == "exportMaterials" && masOption[1] == "false") 
		{
				g_bExportMaterials = FALSE;
		}
		else if (masOption[0] == "exportAnimation" && masOption[1] == "false") 
		{
			g_bExportAnimation = FALSE;
		}
		else if (masOption[0] == "keyframeAnimation" && masOption[1] == "false") 
		{
			g_bKeyframeAnimation = FALSE;
		}
		else if (masOption[0] == "animateEverything" && masOption[1] == "false") 
		{
			g_bAnimateEverything = FALSE;
		}
		else if (masOption[0] == "frameStep") 
		{
			g_iFrameStep = masOption[1].asInt();
		}
		else if (masOption[0] == "flipU" && masOption[1] == "false") 
		{
			g_fFlipU *= -1.0f;
		}
		else if (masOption[0] == "flipV" && masOption[1] == "false") 
		{
			g_fFlipV *= -1.0f;
		}
		else if (masOption[0] == "relTexFilename" && masOption[1] == "true") 
		{
			g_bRelativeTexFile = TRUE;
		}
	}

e_Exit:
	if (FAILED(hr))
		cout << "ParseOptions(): An error occured.\n";
	return hr;
}



//-----------------------------------------------------------------------------
// HELPER FUNCTIONS
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Name: AddScene()
// Desc: Loads a Maya 4.0 scene and saves it to an X file object
//-----------------------------------------------------------------------------

HRESULT	AddScene
		(
			LPDIRECTXFILESAVEOBJECT	pxofSave
		)
{
	HRESULT	hr	= S_OK;
	MStatus stat = MS::kSuccess;

	LPDIRECTXFILEDATA pAnimSetObject       = NULL;
	LPDIRECTXFILEDATA pRootFrameObject     = NULL;
	LPDIRECTXFILEDATA pRootTransformObject = NULL;

	DWORD nAnims, iAnim;
	SAnim* aAnims = NULL;

	float rgfIdentity[16] = {1.0f, 0.0f, 0.0f, 0.0f,
							 0.0f, 1.0f, 0.0f, 0.0f,
							 0.0f, 0.0f, 1.0f, 0.0f,
							 0.0f, 0.0f, 0.0f, 1.0f};


	MItDag itWeightGeometryFilters(MItDag::kDepthFirst, MFn::kWeightGeometryFilt);
	MItDag itIkEffectors(MItDag::kDepthFirst, MFn::kIkEffector);

	MItDag itDag(MItDag::kBreadthFirst, MFn::kTransform, &stat);
	if (!stat)
	{
		hr = E_FAIL;
		goto e_Exit;
	}

	// create the SCENE_ROOT Frame
	hr = pxofSave->CreateDataObject(TID_D3DRMFrame, SCENE_ROOT, NULL, 0, NULL, &pRootFrameObject);
	if (FAILED(hr))
		goto e_Exit;

	// create the SCENE_ROOT FrameTransformMatrix (set to identity)
	hr = pxofSave->CreateDataObject(TID_D3DRMFrameTransformMatrix, NULL, NULL, 16 * sizeof(float), rgfIdentity, &pRootTransformObject);
	if (FAILED(hr))
		goto e_Exit;

	hr = pRootFrameObject->AddDataObject(pRootTransformObject);
	if (FAILED(hr))
		goto e_Exit;

	cout << endl << "Loading scene..." << endl;

	for (; !itDag.isDone() && itDag.depth() <= 1; itDag.next())
	{
		MDagPath dagPath;

		stat = itDag.getPath(dagPath);
		if (!stat)
		{
			cout << "WARNING: Ignoring shape because could not get path to shape." << endl;
			continue;
		}

		if (!g_bIntermediateObjects_c && MFnDagNode(dagPath).isIntermediateObject()) 
			continue;

		hr = AddShape(dagPath, true, pRootFrameObject, pxofSave);
		if (FAILED(hr))
			cout << "AddScene(): Could not add top level shape." << endl;
	}

	cout << "Writing scene data (this can take a few minutes)...";
	cout.flush();

	hr = pxofSave->SaveData(pRootFrameObject);
	if (FAILED(hr))
		goto e_Exit;

	cout << "DONE!" << endl;
	cout.flush();

	if (!g_bExportAnimation) 
		goto e_Exit;

	hr = pxofSave->CreateDataObject(TID_D3DRMAnimationSet, NULL, NULL, 0, NULL, &pAnimSetObject);
	if (FAILED(hr))
		goto e_Exit;

	cout << "\nLoading animations...\n";
	cout.flush();

	if (g_bKeyframeAnimation)
		hr = LoadKeyframedAnims(&nAnims, &aAnims);
	else
		hr = LoadFixedStepAnims(&nAnims, &aAnims);
	if (FAILED(hr))
		goto e_Exit;

	cout << "\t" << nAnims << " anim(s) loaded." << endl;

	cout << "Writing animations (this can take a few minutes)...";
	cout.flush();

	for (iAnim = 0; iAnim < nAnims; iAnim++)
	{
		hr = AddAnim(aAnims + iAnim, pAnimSetObject, pxofSave);
		if (FAILED(hr))
			continue;
	}

	hr = pxofSave->SaveData(pAnimSetObject);
	if (FAILED(hr))
		goto e_Exit;

	cout << "DONE!\n";
	cout.flush();

e_Exit:

	if (pRootFrameObject)
		pRootFrameObject->Release();

	if (pRootTransformObject)
		pRootTransformObject->Release();

	if (pAnimSetObject)
		pAnimSetObject->Release();


	cout << "\nREMARKS:" << endl;

	for (itWeightGeometryFilters.reset(); !itWeightGeometryFilters.isDone(); itWeightGeometryFilters.next()) 
	{
		if (itWeightGeometryFilters.item().hasFn(MFn::kJointCluster))
			continue;	// rigid skinning is OK

		cout << " - Unsupported skinning detected." << endl;
		break;
	}

	if (!itIkEffectors.isDone())
		cout << " - IkEffector nodes were found.  This will probably affect skinning info." << endl;

	if ((g_bExportNURBS && 1 == g_dwNURBSTess) || (g_bExportSubd && 0 != g_dwSubdTess))
		cout << " - When a NURBS or SubD surface is tesselated, it loses its material and skinning info." << endl;

	cout << " - Post-skinning operations will lead to bad results.\n";
	cout << "   (For example, vertices added after skinning will have no skin-weights.)\n";

	
	delete[] aAnims;

	if (FAILED(hr))
		cout << "AddScene(): An error occured.\n";
	
	return hr;
}



//-----------------------------------------------------------------------------
// Name: LoadTransformKey()
// Desc: Loads the transform info from a specified DAG path
//-----------------------------------------------------------------------------

HRESULT LoadTransformKey
		(
			MDagPath&	mdpTransform,
			SKey*		pKey
		)
{
	HRESULT hr = S_OK;
	MStatus stat;
	MFnTransform fnTransform;
	MFnIkJoint fnJoint;
	DOUBLE pdScale[3], pdShear[3], pdRotationQuat[4];
	MEulerRotation merRotation;
	MVector mvecTranslation, mvecScaleTranslation, mvecRotationTranslation;
	MPoint mptScalePivot, mptRotationPivot;
	MQuaternion mqRotationOrientation;

	if (!mdpTransform.hasFn(MFn::kTransform))
	{
		hr = E_FAIL;
		goto e_Exit;
	}

	if (!fnTransform.setObject(mdpTransform.node()))
	{
		hr = E_FAIL;
		goto e_Exit;
	}


	// Scale
	if (!fnTransform.getScale(pdScale))
	{
		hr = E_FAIL;
		goto e_Exit;
	}

	pKey->m_f3Scale[0] = (FLOAT)pdScale[0];
	pKey->m_f3Scale[1] = (FLOAT)pdScale[1];
	pKey->m_f3Scale[2] = (FLOAT)pdScale[2];

	// Shear
	if (!fnTransform.getShear(pdShear))
	{
		hr = E_FAIL;
		goto e_Exit;
	}

	// Scale Pivot Point
	mptScalePivot = fnTransform.scalePivot(MSpace::kTransform, &stat);
	if (!stat)
	{
		hr = E_FAIL;
		goto e_Exit;
	}

	mptScalePivot.homogenize();

 	// Scale Pivot Translation
	mvecScaleTranslation = fnTransform.scalePivotTranslation(MSpace::kTransform, &stat);
	if (!stat)
	{
		hr = E_FAIL;
		goto e_Exit;
	}

	// Rotation Orientation (quaternion)
	mqRotationOrientation = fnTransform.rotateOrientation(MSpace::kTransform, &stat);
	if (!stat)
	{
		hr = E_FAIL;
		goto e_Exit;
	}

	// Rotation (euler)
	if (!fnTransform.getRotation(merRotation))
	{
		hr = E_FAIL;
		goto e_Exit;
	}

	pKey->m_f3Euler[0] = (FLOAT)(merRotation.x);    // (in radians)
	pKey->m_f3Euler[1] = (FLOAT)(merRotation.y);    // (in radians)
	pKey->m_f3Euler[2] = (FLOAT)(merRotation.z);    // (in radians)

	// Rotation Order (XYZ = 0, YZX = 1, ZXY = 2, XZY = 3, YXZ = 4, ZYX = 5)
	pKey->m_dwEulerOrder = (DWORD)merRotation.order;	

	// Rotation (quaternion)
	if (!fnTransform.getRotationQuaternion(pdRotationQuat[0], pdRotationQuat[1], pdRotationQuat[2], pdRotationQuat[3], MSpace::kTransform))
	{
		hr = E_FAIL;
		goto e_Exit;
	}

	pKey->m_f4Orientation[0] = (FLOAT)pdRotationQuat[0];
	pKey->m_f4Orientation[1] = (FLOAT)pdRotationQuat[1];
	pKey->m_f4Orientation[2] = (FLOAT)pdRotationQuat[2];
	pKey->m_f4Orientation[3] = (FLOAT)pdRotationQuat[3];

	// Rotation Pivot Point
	mptRotationPivot = fnTransform.rotatePivot(MSpace::kTransform, &stat);
	if (!stat)
	{
		hr = E_FAIL;
		goto e_Exit;
	}

	mptRotationPivot.homogenize();

 	// Rotation Pivot Translation
	mvecRotationTranslation = fnTransform.rotatePivotTranslation(MSpace::kTransform, &stat);
	if (!stat)
	{
		hr = E_FAIL;
		goto e_Exit;
	}

	// Translation
	mvecTranslation = fnTransform.translation(MSpace::kTransform, &stat);
	if (!stat)
	{
		hr = E_FAIL;
		goto e_Exit;
	}

	pKey->m_f3Translation[0] = (FLOAT)mvecTranslation.x;
	pKey->m_f3Translation[1] = (FLOAT)mvecTranslation.y;
	pKey->m_f3Translation[2] = (FLOAT)mvecTranslation.z;

	// DOFs
	if (mdpTransform.hasFn(MFn::kJoint))
	{	// get joint's DOFs
		if (!fnJoint.setObject(mdpTransform.node()) ||
		    !fnJoint.getDegreesOfFreedom(pKey->m_pbDOFs[0], pKey->m_pbDOFs[1], pKey->m_pbDOFs[2]))
		{
			hr = E_FAIL;
			goto e_Exit;
		}
	}
	else
	{	// ordinary (non-joint) transforms have all 3 DOFs
		pKey->m_pbDOFs[0] = true;
		pKey->m_pbDOFs[1] = true;
		pKey->m_pbDOFs[2] = true;
	}

#ifdef KAYA
{
	DWORD i;
	MFnBlendShapeDeformer fnBlend;
	MDagPath mdpShape;
	MIntArray maiWeightIndices;
	MItDependencyGraph* pmitdg = NULL;

	// extend the transform to a shape
	mdpShape = mdpTransform;
	stat = mdpShape.extendToShape();	// TODO: this assumes only 1 shape per transform
	if (!stat)
		goto e_ExitKAYA;
	
	// search upstream for blend shape deformer nodes
	if (NULL == (pmitdg = new MItDependencyGraph(mdpShape.node(), MFn::kBlendShape, 
												 MItDependencyGraph::kUpstream, MItDependencyGraph::kDepthFirst, 
												 MItDependencyGraph::kNodeLevel, &stat)))
		goto e_ExitKAYA;

	stat = fnBlend.setObject(pmitdg->thisNode());	// TODO: this assumes only 1 blendShapeDeformer per shape
	if (!stat)
		goto e_ExitKAYA;

	// get the weight indices (this is needed since the array may be sparse)
	stat = fnBlend.weightIndexList(maiWeightIndices);
	if (!stat)
		goto e_ExitKAYA;

	pKey->m_cBlendWts = maiWeightIndices.length();
			 
	if (NULL == (pKey->m_rgfBlendWts = new FLOAT[pKey->m_cBlendWts]))
	{
		hr = E_OUTOFMEMORY;
		goto e_ExitKAYA;
	}

	for (i = 0; i < pKey->m_cBlendWts; i++)
		pKey->m_rgfBlendWts[i] = fnBlend.weight(maiWeightIndices[i]);

e_ExitKAYA:
	delete pmitdg;
}
#endif

e_Exit:
	if (FAILED(hr))
		cout << "LoadTransformKey(): An error occured.\n";

	return hr;
}



//-----------------------------------------------------------------------------
// Name: AddTransform()
// Desc: Adds transform info from a specified DAG path to an XFile Data Object
//-----------------------------------------------------------------------------

HRESULT	AddTransform
		(
			MDagPath&				mdpTransform, 
			LPDIRECTXFILEDATA		pFrameDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		) 
{
	HRESULT hr = S_OK;
	MStatus stat;
	LPDIRECTXFILEDATA pParamsDataObject = NULL;
	LPDIRECTXFILEDATA pMatrixDataObject = NULL;
	PBYTE pbParams, pbParamsData = NULL;
	DWORD cbParams, nParams;
	SKey Key;
	DWORD i, j;		// counters
	MFnTransform fnTransform;
	float *pfMatrix = NULL;

	// FrameTransformMatrix
	if (!mdpTransform.hasFn(MFn::kTransform) ||
        !fnTransform.setObject(mdpTransform))
	{
		hr = E_FAIL;
		goto e_Exit;
	}

	if (NULL == (pfMatrix = new float[16]))
	{
		hr = E_OUTOFMEMORY;
		goto e_Exit;
	}

	for (i = 0; i < 4; i++)
		for (j = 0; j < 4; j++)
			pfMatrix[i * 4 + j] = (float)fnTransform.transformation().asMatrix()[i][j];
	
	if (FAILED(hr = g_Arrays.Add(pfMatrix)))
		goto e_Exit;
	
	if (FAILED(hr = pxofSave->CreateDataObject(TID_D3DRMFrameTransformMatrix, NULL, NULL, 16 * sizeof(float), pfMatrix, &pMatrixDataObject)))
		goto e_Exit;
	
	if (FAILED(hr = pFrameDataObject->AddDataObject(pMatrixDataObject)))
		goto e_Exit;

	// Params
	if (FAILED(hr = LoadTransformKey(mdpTransform, &Key)))
		goto e_Exit;

	nParams = (DWORD)Key.m_pbDOFs[0] + (DWORD)Key.m_pbDOFs[1] + (DWORD)Key.m_pbDOFs[2];

	cbParams = sizeof(DWORD)			// nValues
			 + nParams * sizeof(FLOAT);	// values[nValues]
				 
	if (NULL == (pbParams = pbParamsData = new BYTE[cbParams]))
	{
		hr = E_OUTOFMEMORY;
		goto e_Exit;
	}

	// nValues
	WRITE(DWORD, pbParams, nParams);

	// values
	for (i = 0; i < 3; i++)
		if (Key.m_pbDOFs[i])
			WRITE(FLOAT, pbParams, Key.m_f3Euler[i]);


	if (FAILED(hr = pxofSave->CreateDataObject(TID_D3DRMFloatKeys, NULL, NULL, cbParams, pbParamsData, &pParamsDataObject)))
		goto e_Exit;

	// add the data objects to frame data object
	if (g_bExportPose)
	{
		if (FAILED(hr = pFrameDataObject->AddDataObject(pParamsDataObject)))
			goto e_Exit;
	}

#ifdef KAYA
	else if (0 < Key.m_cBlendWts)
	{
		LPDIRECTXFILEDATA pParamsDataObject = NULL;
		PBYTE pbParams, pbParamsData = NULL;
		DWORD cbParams = sizeof(DWORD)			// nValues
					   + Key.m_cBlendWts * sizeof(FLOAT);	// values[nValues]
					 
		if (NULL == (pbParams = pbParamsData = new BYTE[cbParams]))
		{
			hr = E_OUTOFMEMORY;
			goto e_ExitKAYA;
		}

		// nValues
		WRITE(DWORD, pbParams, Key.m_cBlendWts);

		// values
		for (i = 0; i < Key.m_cBlendWts; i++)
			WRITE(FLOAT, pbParams, Key.m_rgfBlendWts[i]);

		hr = pxofSave->CreateDataObject(TID_D3DRMFloatKeys, NULL, NULL, cbParams, pbParamsData, &pParamsDataObject);
		if (FAILED(hr))
			goto e_ExitKAYA;

		hr = pFrameDataObject->AddDataObject(pParamsDataObject);
		if (FAILED(hr))
			goto e_ExitKAYA;

e_ExitKAYA:
		SAFE_DELETE_ARRAY(pbParamsData);
		SAFE_RELEASE(pParamsDataObject);

		if (FAILED(hr))
			goto e_Exit;
	}
#endif

e_Exit:
	SAFE_DELETE_ARRAY(pbParamsData);

	SAFE_RELEASE(pMatrixDataObject);
    SAFE_RELEASE(pParamsDataObject);

    if (FAILED(hr))
        cout << "AddTransform(): An error occured.\n";

	return hr;
}



//-----------------------------------------------------------------------------
// Name: AddAnim()
// Desc: Adds an animation to an XFile Data Object
//-----------------------------------------------------------------------------

HRESULT	AddAnim
		(
			SAnim*					pAnim, 
			LPDIRECTXFILEDATA		pAnimSetObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		) 
{
	HRESULT hr = S_OK;
	LPDIRECTXFILEDATA pAnimDataObject = NULL;
	LPDIRECTXFILEDATA pQuaternionKeyDataObject = NULL;
	LPDIRECTXFILEDATA pScaleKeyDataObject = NULL;
	LPDIRECTXFILEDATA pPositionKeyDataObject = NULL;
	LPDIRECTXFILEDATA pParamsKeyDataObject = NULL;
#ifdef KAYA
	LPDIRECTXFILEDATA pBlendKeyDataObject = NULL;
	PBYTE pbBlendKey, pbBlendKeyData = NULL;
	DWORD cbBlendKey;	// calculate this based on number of blend wts
	DWORD nBlends;		// number of blends
#endif
	PBYTE pbPositionKey, pbPositionKeyData = NULL;
	PBYTE pbScaleKey, pbScaleKeyData = NULL;
	PBYTE pbQuaternionKey, pbQuaternionKeyData = NULL;
	PBYTE pbParamsKey, pbParamsKeyData = NULL;
	DWORD iKey, i;	// counters
	DWORD cbQuaternionKey = sizeof(DWORD)	// keyType
						  + sizeof(DWORD)	// nKeys
						  + pAnim->m_cKeys	// keys[nKeys]
							* (sizeof(DWORD) + sizeof(DWORD) + 4 * sizeof(float));	
	DWORD cbScaleKey = sizeof(DWORD)		// keyType
					 + sizeof(DWORD)		// nKeys
					 + pAnim->m_cKeys		// keys[nKeys]
					   * (sizeof(DWORD) + sizeof(DWORD) + 3 * sizeof(float));		
	DWORD cbPositionKey = sizeof(DWORD)		// keyType
						+ sizeof(DWORD)		// nKeys
						+ pAnim->m_cKeys	// keys[nKeys]
						  * (sizeof(DWORD) + sizeof(DWORD) + 3 * sizeof(float));
	DWORD cbParamsKey;	// calculate this based on nParams
	DWORD nParams;		// number of params

	nParams = 0;
	if (0 < pAnim->m_cKeys)
	{
		nParams = (DWORD)pAnim->m_pKeys[0].m_pbDOFs[0] + 
				+ (DWORD)pAnim->m_pKeys[0].m_pbDOFs[1] + 
				+ (DWORD)pAnim->m_pKeys[0].m_pbDOFs[2];
	}

	cbParamsKey = sizeof(DWORD)		// keyType
				+ sizeof(DWORD)		// nKeys
				+ pAnim->m_cKeys	// keys[nKeys]
				  * (sizeof(DWORD) + sizeof(DWORD) + nParams * sizeof(FLOAT));	
	
	if (FAILED(hr = pxofSave->CreateDataObject(TID_D3DRMAnimation, NULL, NULL, 0, NULL, &pAnimDataObject)))
		goto e_Exit;

    if (NULL == (pbPositionKey = pbPositionKeyData = new BYTE[cbPositionKey]) ||
	    NULL == (pbScaleKey = pbScaleKeyData = new BYTE[cbScaleKey]) ||
	    NULL == (pbQuaternionKey = pbQuaternionKeyData = new BYTE[cbQuaternionKey]) ||
        NULL == (pbParamsKey = pbParamsKeyData = new BYTE[cbParamsKey]))
	{
		hr = E_OUTOFMEMORY;
		goto e_Exit;
	}

#ifdef KAYA
	nBlends = 0;
	if (0 < pAnim->m_cKeys) 
		nBlends = pAnim->m_pKeys[0].m_cBlendWts;

	cbBlendKey = sizeof(DWORD)		// keyType
				+ sizeof(DWORD)		// nKeys
				+ pAnim->m_cKeys	// keys[nKeys]
				  * (sizeof(DWORD) + sizeof(DWORD) + nBlends * sizeof(FLOAT));	

	if (NULL == (pbBlendKey = pbBlendKeyData = new BYTE[cbBlendKey]))
	{
		hr = E_OUTOFMEMORY;
		goto e_Exit;
	}

	WRITE(DWORD, pbBlendKey, 15);	// 15 - blend/params key type
	WRITE(DWORD, pbBlendKey, ((DWORD)pAnim->m_cKeys));	// nKeys
#endif

	// keyType
	WRITE(DWORD, pbQuaternionKey, 0);	// 0 - quaternion key type
	WRITE(DWORD, pbScaleKey, 1);		// 1 - scale key type
	WRITE(DWORD, pbPositionKey, 2);		// 2 - translation key type
	WRITE(DWORD, pbParamsKey, 15);		// 15 - params key type

	// nKeys
	WRITE(DWORD, pbQuaternionKey, ((DWORD)pAnim->m_cKeys));
	WRITE(DWORD, pbScaleKey, ((DWORD)pAnim->m_cKeys));
	WRITE(DWORD, pbPositionKey, ((DWORD)pAnim->m_cKeys));
	WRITE(DWORD, pbParamsKey, ((DWORD)pAnim->m_cKeys));

	// keys[nKeys]
	for (iKey = 0; iKey < pAnim->m_cKeys; iKey++) 
	{
		// time
		WRITE(DWORD, pbQuaternionKey, ((DWORD)pAnim->m_pKeys[iKey].m_iFrame));
		WRITE(DWORD, pbScaleKey, ((DWORD)pAnim->m_pKeys[iKey].m_iFrame));
		WRITE(DWORD, pbPositionKey, ((DWORD)pAnim->m_pKeys[iKey].m_iFrame));
		WRITE(DWORD, pbParamsKey, ((DWORD)pAnim->m_pKeys[iKey].m_iFrame));

		// nValues
		WRITE(DWORD, pbQuaternionKey, 4);
		WRITE(DWORD, pbScaleKey, 3);
		WRITE(DWORD, pbPositionKey, 3);
		WRITE(DWORD, pbParamsKey, nParams);

		// values
		for (i = 0; i < 4; i++)
			WRITE(FLOAT, pbQuaternionKey, pAnim->m_pKeys[iKey].m_f4Orientation[i]);

		for (i = 0; i < 3; i++)
			WRITE(FLOAT, pbScaleKey, pAnim->m_pKeys[iKey].m_f3Scale[i]);
	
		for (i = 0; i < 3; i++)
			WRITE(FLOAT, pbPositionKey, pAnim->m_pKeys[iKey].m_f3Translation[i]);

		for (i = 0; i < 3; i++)
			if (pAnim->m_pKeys[iKey].m_pbDOFs[i])
				WRITE(FLOAT, pbParamsKey, pAnim->m_pKeys[iKey].m_f3Euler[i]);
#ifdef KAYA
		WRITE(DWORD, pbBlendKey, ((DWORD)pAnim->m_pKeys[iKey].m_iFrame));	// time
		WRITE(DWORD, pbBlendKey, nBlends);									// nValues
		for (i = 0; i < nBlends; i++)										// values
			WRITE(FLOAT, pbBlendKey, pAnim->m_pKeys[iKey].m_rgfBlendWts[i]);
#endif
	}

	if (FAILED(hr = pxofSave->CreateDataObject(TID_D3DRMAnimationKey, NULL, NULL, cbQuaternionKey, pbQuaternionKeyData, &pQuaternionKeyDataObject)) ||
	    FAILED(hr = pxofSave->CreateDataObject(TID_D3DRMAnimationKey, NULL, NULL, cbScaleKey, pbScaleKeyData, &pScaleKeyDataObject)) ||
        FAILED(hr = pxofSave->CreateDataObject(TID_D3DRMAnimationKey, NULL, NULL, cbPositionKey, pbPositionKeyData, &pPositionKeyDataObject)) ||
	    FAILED(hr = pxofSave->CreateDataObject(TID_D3DRMAnimationKey, NULL, NULL, cbParamsKey, pbParamsKeyData, &pParamsKeyDataObject)))
		goto e_Exit;

	if (FAILED(hr = pAnimDataObject->AddDataReference(pAnim->m_szName, NULL)))
		goto e_Exit;

	if (FAILED(hr = pAnimDataObject->AddDataObject(pQuaternionKeyDataObject)) ||
	    FAILED(hr = pAnimDataObject->AddDataObject(pScaleKeyDataObject)) ||
	    FAILED(hr = pAnimDataObject->AddDataObject(pPositionKeyDataObject)))
		goto e_Exit;

	if (g_bExportPose)
	{
		if (FAILED(hr = pAnimDataObject->AddDataObject(pParamsKeyDataObject)))
			goto e_Exit;
	}
#ifdef KAYA
	else if (0 < nBlends)
	{
		if (FAILED(hr = pxofSave->CreateDataObject(TID_D3DRMAnimationKey, NULL, NULL, cbBlendKey, pbBlendKeyData, &pBlendKeyDataObject)))
			goto e_Exit;
		
		if (FAILED(hr = pAnimDataObject->AddDataObject(pBlendKeyDataObject)))
			goto e_Exit;
	}
#endif
	
	if (FAILED(hr = pAnimSetObject->AddDataObject(pAnimDataObject)))
		goto e_Exit;

e_Exit:

#ifdef KAYA
	SAFE_DELETE_ARRAY(pbBlendKeyData);
	SAFE_RELEASE(pBlendKeyDataObject);
#endif

    SAFE_DELETE_ARRAY(pbQuaternionKeyData);
	SAFE_DELETE_ARRAY(pbScaleKeyData);
	SAFE_DELETE_ARRAY(pbPositionKeyData);
	SAFE_DELETE_ARRAY(pbParamsKeyData);

	SAFE_RELEASE(pQuaternionKeyDataObject);
	SAFE_RELEASE(pScaleKeyDataObject);
	SAFE_RELEASE(pPositionKeyDataObject);
	SAFE_RELEASE(pParamsKeyDataObject);
	SAFE_RELEASE(pAnimDataObject);

    if (FAILED(hr))
        cout << "AddAnim(): There was an error.\n";

    return hr;
}



//-----------------------------------------------------------------------------
// Name: ReduceNormals()
// Desc: Removes redundant normals from a shape's lump of normals.
//-----------------------------------------------------------------------------

HRESULT	ReduceNormals
		(
			SShape*		pShape,
            DWORD*      piRemap
		) 
{
	HRESULT hr = S_OK;
    DWORD *piRemap1 = NULL;
    DWORD *piRemap2 = NULL;
    DWORD i, j;
    FLOAT f;
    DWORD dw;
    BOOL bSorted, bSwap;
    DWORD cReduced;


    if (NULL == (piRemap1 = new DWORD[pShape->m_cNormals]) ||
        NULL == (piRemap2 = new DWORD[pShape->m_cNormals]))
    {
        hr = E_OUTOFMEMORY;
        goto EXIT;
    }

    for (i = 0; i < pShape->m_cNormals; i++)
        piRemap1[i] = i;

    // sort (component-major) the normals in-place
    // TODO: use a sort faster than Bubble Sort
    do
    {
        for (bSorted = TRUE, i = 0; i + 1 < pShape->m_cNormals; i++)
        {
            for (bSwap = FALSE, j = 0; j < 3; j++)
            {
                if (pShape->m_pNormals[i][j] > pShape->m_pNormals[i + 1][j])
                {
                    bSwap = TRUE;
                    break;
                }

                if (pShape->m_pNormals[i][j] < pShape->m_pNormals[i + 1][j])
                    break;
            }

            if (!bSwap)
                continue;

            for (j = 0; j < 3; j++)
            {
                f = pShape->m_pNormals[i][j];
                pShape->m_pNormals[i][j] = pShape->m_pNormals[i + 1][j];
                pShape->m_pNormals[i + 1][j] = f;
            }

            dw = piRemap1[i];
            piRemap1[i] = piRemap1[i + 1];
            piRemap1[i + 1] = dw;

            bSorted = FALSE;
        }
    } while (!bSorted);

    for (i = 0; i < pShape->m_cNormals; i++)
        piRemap2[piRemap1[i]] = i;

    // reduce the sorted normals in-place
    for (cReduced = 0, i = 0; i < pShape->m_cNormals; cReduced++)
    {
        pShape->m_pNormals[cReduced][0] = pShape->m_pNormals[i][0];
        pShape->m_pNormals[cReduced][1] = pShape->m_pNormals[i][1];
        pShape->m_pNormals[cReduced][2] = pShape->m_pNormals[i][2];

        do
        {
            piRemap1[i] = cReduced;
            i++;
        } while (i < pShape->m_cNormals && 
                 pShape->m_pNormals[cReduced][0] == pShape->m_pNormals[i][0] &&
                 pShape->m_pNormals[cReduced][1] == pShape->m_pNormals[i][1] &&
                 pShape->m_pNormals[cReduced][2] == pShape->m_pNormals[i][2]);
    }

    // compose the two remaps
    for (i = 0; i < pShape->m_cNormals; i++)
        piRemap[i] = piRemap1[piRemap2[i]];

    // save the number of normals
    pShape->m_cNormals = cReduced;

EXIT:
    // clean up
    SAFE_DELETE_ARRAY(piRemap1);
    SAFE_DELETE_ARRAY(piRemap2);

    if (FAILED(hr))
        cout << "ReduceNormals(): An error occured.\n";

    return hr;
}

       
//-----------------------------------------------------------------------------
// Name: ReduceUVs()
// Desc: Removes redundant UVs from a shape's lump of normals.
//-----------------------------------------------------------------------------

HRESULT	ReduceUVs
		(
			SShape*		pShape,
            DWORD*      piRemap
		) 
{
	HRESULT hr = S_OK;
    DWORD *piRemap1 = NULL;
    DWORD *piRemap2 = NULL;
    DWORD i, j;
    FLOAT f;
    DWORD dw;
    BOOL bSorted, bSwap;
    DWORD cReduced;

    // allocate memory for temporary remap arrays
    if (NULL == (piRemap1 = new DWORD[pShape->m_cUVs]) ||
        NULL == (piRemap2 = new DWORD[pShape->m_cUVs]))
    {
        hr = E_OUTOFMEMORY;
        goto EXIT;
    }

    // initialize sort remap
    for (i = 0; i < pShape->m_cUVs; i++)
        piRemap1[i] = i;

    // sort (component-major) the normals in-place
    // TODO: use a sort faster than Bubble Sort
    do
    {
        for (bSorted = TRUE, i = 0; i + 1 < pShape->m_cUVs; i++)
        {
            for (bSwap = FALSE, j = 0; j < 2; j++)
            {
                if (pShape->m_pUVs[i][j] > pShape->m_pUVs[i + 1][j])
                {
                    bSwap = TRUE;
                    break;
                }

                if (pShape->m_pUVs[i][j] < pShape->m_pUVs[i + 1][j])
                    break;
            }

            if (!bSwap)
                continue;

            for (j = 0; j < 2; j++)
            {
                f = pShape->m_pUVs[i][j];
                pShape->m_pUVs[i][j] = pShape->m_pUVs[i + 1][j];
                pShape->m_pUVs[i + 1][j] = f;
            }

            dw = piRemap1[i];
            piRemap1[i] = piRemap1[i + 1];
            piRemap1[i + 1] = dw;

            bSorted = FALSE;
        }
    } while (!bSorted);

    // save the reversed sort remap
    for (i = 0; i < pShape->m_cUVs; i++)
        piRemap2[piRemap1[i]] = i;

    // reduce the sorted normals in-place
    for (cReduced = 0, i = 0; i < pShape->m_cUVs; cReduced++)
    {
        pShape->m_pUVs[cReduced][0] = pShape->m_pUVs[i][0];
        pShape->m_pUVs[cReduced][1] = pShape->m_pUVs[i][1];

        do
        {
            piRemap1[i] = cReduced;
            i++;
        } while (i < pShape->m_cUVs && 
                 pShape->m_pUVs[cReduced][0] == pShape->m_pUVs[i][0] &&
                 pShape->m_pUVs[cReduced][1] == pShape->m_pUVs[i][1]);
    }

    // compose the two remaps
    for (i = 0; i < pShape->m_cUVs; i++)
        piRemap[i] = piRemap1[piRemap2[i]]; // piRemap[i] = the new index of i

    // save the number of normals
    pShape->m_cUVs = cReduced;

EXIT:
    // clean up
    SAFE_DELETE_ARRAY(piRemap1);
    SAFE_DELETE_ARRAY(piRemap2);

    if (FAILED(hr))
        cout << "ReduceUVs(): An error occured.\n";

    return hr;
}

       
//-----------------------------------------------------------------------------
// Name: InterpolateMissingSkinWeights()
// Desc: Interpolates skin weights of nearest vertices for those with
//       missing skin weights.
//-----------------------------------------------------------------------------

HRESULT	InterpolateMissingSkinWeights
		(
			SShape*		pShape
		) 
{
	HRESULT hr = S_OK;

	DWORD i, j, k;
	DWORD iBad, iHorrible;
    DWORD cMaxNeighbors, cGoodNeighbors;
	PDWORD rgcNeighbors = NULL;
	PDWORD* rgrgiNeighbors = NULL;
	PDWORD rgiIndices = NULL;
	PFLOAT rgfWeights = NULL;
	PFLOAT rgfVertWtSums = NULL;
	PFLOAT rgfBoneVertWts = NULL;
    PBOOL rgbSeen = NULL;
    PFLOAT rgfGoodDistSq = NULL;
    PDWORD rgiGoodNeighbors = NULL;
	FLOAT fWeight, fDelta, fDistSq;
    FLOAT fTotalGoodDistSq;

	if (NULL == (rgbSeen = new BOOL[pShape->m_cPoints]) ||
		NULL == (rgcNeighbors = new DWORD[pShape->m_cPoints]) ||
		NULL == (rgrgiNeighbors = new PDWORD[pShape->m_cPoints]) ||
		NULL == (rgiIndices = new DWORD[pShape->m_cPoints]) ||
		NULL == (rgfWeights = new FLOAT[pShape->m_cPoints]) ||
		NULL == (rgfBoneVertWts = new FLOAT[pShape->m_cPoints * pShape->m_cBones]) ||
		NULL == (rgfVertWtSums = new FLOAT[pShape->m_cPoints]))
	{
		hr = E_OUTOFMEMORY;
		goto e_Exit;
	}
	
	ZeroMemory(rgcNeighbors, pShape->m_cPoints * sizeof(DWORD));
	ZeroMemory(rgrgiNeighbors, pShape->m_cPoints * sizeof(PDWORD));
	ZeroMemory(rgfBoneVertWts, pShape->m_cPoints * pShape->m_cBones * sizeof(FLOAT));
	ZeroMemory(rgfVertWtSums, pShape->m_cPoints * sizeof(FLOAT));
    ZeroMemory(rgbSeen, pShape->m_cPoints * sizeof(BOOL));

	// compute the table of bone/vertex weights as well as the vertex weight sums
	for (i = 0; i < pShape->m_cBones; i++)
	{
		for (j = 0; j < pShape->m_pBones[i].m_cWeights; j++)
		{
			rgfBoneVertWts[i * pShape->m_cPoints + pShape->m_pBones[i].m_piPoints[j]] += pShape->m_pBones[i].m_pfWeights[j];
			rgfVertWtSums[pShape->m_pBones[i].m_piPoints[j]] += pShape->m_pBones[i].m_pfWeights[j];
		}
	}

    // calculate worst-case number of neighbors
	for (i = 0; i < pShape->m_cFaces; i++)
		for (j = 0; j < pShape->m_pFaces[i].m_cIndices; j++)
			rgcNeighbors[pShape->m_pReps[pShape->m_pFaces[i].m_pIndices[j]].m_iFirst] += 2;

	for (cMaxNeighbors = 0, i = 0; i < pShape->m_cPoints; i++)
	{
		if (NULL == (rgrgiNeighbors[i] = new DWORD[rgcNeighbors[i]]))
		{
			hr = E_OUTOFMEMORY;
			goto e_Exit;
		}

        if (cMaxNeighbors < rgcNeighbors[i])
            cMaxNeighbors = rgcNeighbors[i];

        // reset counts
        rgcNeighbors[i] = 0;
	}

    if (NULL == (rgfGoodDistSq = new FLOAT[cMaxNeighbors]) ||
        NULL == (rgiGoodNeighbors = new DWORD[cMaxNeighbors]))
    {
        hr = E_OUTOFMEMORY;
		goto e_Exit;
	}


	for (i = 0; i < pShape->m_cFaces; i++)
	{
		for (j = 0; j < pShape->m_pFaces[i].m_cIndices; j++)
		{
			k = pShape->m_pReps[pShape->m_pFaces[i].m_pIndices[j]].m_iFirst;
			rgrgiNeighbors[k][0 + rgcNeighbors[k]] = pShape->m_pReps[pShape->m_pFaces[i].m_pIndices[(pShape->m_pFaces[i].m_cIndices + j - 1) % pShape->m_pFaces[i].m_cIndices]].m_iFirst;
			rgrgiNeighbors[k][1 + rgcNeighbors[k]] = pShape->m_pReps[pShape->m_pFaces[i].m_pIndices[(pShape->m_pFaces[i].m_cIndices + j + 1) % pShape->m_pFaces[i].m_cIndices]].m_iFirst;
			rgcNeighbors[k] += 2;
		}
	}

	for (i = 0; i < pShape->m_cPoints; i++)
		rgiIndices[i] = i;
									// "good" vertices are those which have non-zero sum weights
	iBad = 0;						// "bad" vertices are those which have 0 sum weights but have a "good" neighbor
	iHorrible = pShape->m_cPoints;	// "horrible" vertices are those which have 0 sum weights and have no "good" neighbor

	// arrange vertices in the order of GOOD...BAD...HORRIBLE, and work from left to right
	while (iBad < pShape->m_cPoints)
	{
		if (iBad == iHorrible)
			iHorrible = pShape->m_cPoints;
		
		i = rgiIndices[iBad];

		if (0.0f != rgfVertWtSums[i])
		{
			iBad++;
			continue;
		}


		// find the nearest "good" neighbor that has weights
		for (fTotalGoodDistSq = 0.0f, cGoodNeighbors = 0, j = 0; j < rgcNeighbors[i]; j++)
		{
            if (rgbSeen[j])
                continue;

            rgbSeen[j] = TRUE;

			if (0.0f == rgfVertWtSums[rgrgiNeighbors[i][j]])
				continue;

			for (fDistSq = 0.0f, k = 0; k < 3; k++)
			{
				fDelta = pShape->m_pPoints[i][k] - pShape->m_pPoints[rgrgiNeighbors[i][j]][k];
				fDistSq += fDelta * fDelta;
			}

//            fDistSq = (FLOAT)sqrt(fDistSq);

            fTotalGoodDistSq += fDistSq;

            rgfGoodDistSq[cGoodNeighbors] = fDistSq;
            rgiGoodNeighbors[cGoodNeighbors] = j;

            cGoodNeighbors++;
		}

        // reset the seen-flags
        for (j = 0; j < rgcNeighbors[i]; j++)
            rgbSeen[j] = FALSE;


		if (0 == cGoodNeighbors)
		{
			iHorrible--;
			rgiIndices[iBad] = rgiIndices[iHorrible];
			rgiIndices[iHorrible] = i;
			continue;
		}

        rgfVertWtSums[i] = 0.0f;
		for (j = 0; j < pShape->m_cBones; j++)
			rgfBoneVertWts[j * pShape->m_cPoints + i] = 0;

        for (j = 0; j < cGoodNeighbors; j++)
        {
//            fWeight = 1.0f - rgfGoodDistSq[j] / ((FLOAT)(cGoodNeighbors - 1) * fTotalGoodDistSq);
            fWeight = 1.0f / (FLOAT)(cGoodNeighbors);

		    for (k = 0; k < pShape->m_cBones; k++)
			    rgfBoneVertWts[k * pShape->m_cPoints + i] = fWeight * rgfBoneVertWts[k * pShape->m_cPoints + rgiGoodNeighbors[j]];
		    rgfVertWtSums[i] += fWeight * rgfVertWtSums[rgiGoodNeighbors[j]];
        }

		iBad++;
	}

	for (i = 0; i < pShape->m_cBones; i++)
	{
		DWORD cWeights;

		for (cWeights = 0, j = 0; j < pShape->m_cPoints; j++)
		{
			if (0.0f != rgfBoneVertWts[i * pShape->m_cPoints + j])
			{
				rgfWeights[cWeights] = rgfBoneVertWts[i * pShape->m_cPoints + j];
				rgiIndices[cWeights] = j;
				cWeights++;
			}
		}

		delete[] pShape->m_pBones[i].m_pfWeights;
		delete[] pShape->m_pBones[i].m_piPoints;
		pShape->m_pBones[i].m_pfWeights = NULL;
		pShape->m_pBones[i].m_piPoints = NULL;

		if (NULL == (pShape->m_pBones[i].m_pfWeights = new FLOAT[cWeights]) ||
			NULL == (pShape->m_pBones[i].m_piPoints = new DWORD[cWeights]))
		{
			hr = E_OUTOFMEMORY;
			goto e_Exit;
		}

		memcpy(pShape->m_pBones[i].m_pfWeights, rgfWeights, cWeights * sizeof(FLOAT));
		memcpy(pShape->m_pBones[i].m_piPoints, rgiIndices, cWeights * sizeof(DWORD));
		pShape->m_pBones[i].m_cWeights = cWeights;
	}


e_Exit:
    SAFE_DELETE_ARRAY(rgbSeen);
    SAFE_DELETE_ARRAY(rgiGoodNeighbors);
    SAFE_DELETE_ARRAY(rgfGoodDistSq);
	SAFE_DELETE_ARRAY(rgiIndices);
	SAFE_DELETE_ARRAY(rgfWeights);
	SAFE_DELETE_ARRAY(rgfVertWtSums);
	SAFE_DELETE_ARRAY(rgfBoneVertWts);
	SAFE_DELETE_ARRAY(rgcNeighbors);
	if (NULL != rgrgiNeighbors)
		for (i = 0; i < pShape->m_cPoints; i++)
			SAFE_DELETE_ARRAY(rgrgiNeighbors[i]);
	SAFE_DELETE_ARRAY(rgrgiNeighbors);


	return hr;
}



//-----------------------------------------------------------------------------
// Name: LoadMesh()
// Desc: Big function that loads all the mesh data from a given DAG path
//-----------------------------------------------------------------------------

HRESULT	LoadMesh
		(
			MDagPath&	mdpTransform, 
			MDagPath&	mdpMesh,
			SShape*		pShape
		) 
{
	HRESULT	hr = S_OK;
	MStatus	stat = MStatus::kSuccess;


	DWORD iVert, iRep, iNorm, iUV, iGroup, iFace, iIndex, iBone, iBoneRemap;
	DWORD cRepsMax, cBonesMax = 1;
	DWORD iInstance;

	MFnMesh fnMesh;

	MMatrix	mmWorldTransform;
	bool bIkSnap, bIkSolve;
	bool bVisibility, bLodVisibility, bOverrideEnabled, bOverrideVisibility;

	// geometry info
	MFloatPointArray mfpaPoints;
	MFloatVectorArray mfvaNormals;
	MFloatArray mfaUs, mfaVs;
    DWORD *piNormalRemap = NULL;    // remap array for normals (required because we remove redundant normals)
    DWORD *piUVRemap = NULL;        // remap array for UVs (required because we remove redundant UVs)

	// material info
	MObjectArray maoShaders;
	MIntArray maiPolyShaderMap;

	MItMeshPolygon itPoly(MObject::kNullObj);
	MItDag itBones(MItDag::kDepthFirst, MFn::kJoint);

	// skinning info
	bool bSmoothSkinning, bRigidSkinning;
	MItDependencyNodes itDepNodes;
	MFnSkinCluster fnSkin;
	MTransformationMatrix* amtmBonePositions = NULL;
	MFnWeightGeometryFilter fnCluster;
	bool** aabBonePointTable = NULL;	// (aabBonePointTable[i][j] == true) iff bone i influences point j
	DWORD* acBonesPerPoint = NULL;		// acBonesPerPoint[j] = num bones that influence point j (used for smooth skinning)
	MObjectArray maoBones, maoWorldBindMatrices, maoLocalBindMatrices;
	MIntArray maiBoneRemap;
	// the following are used per bone
	MFnTransform fnBone;
	MObject moComponents, moWorldBindMatrix, moLocalBindMatrix;
	MPlug mpToBindPose;
	MFnMatrixData fnMatrix;
	char* szBone = NULL;
	char* pc;
	float* afWeights = NULL;
	DWORD* aiPoints = NULL;
	bool* abBonePointArray = NULL;
	DWORD cWeights;
	DWORD iBoneIdx, iWeight;

	// initialize mesh
	pShape->m_kType = SShape::UNKNOWN;
	pShape->m_cGroups = 0;
	pShape->m_pGroups = NULL;
	pShape->m_cPoints = 0;
	pShape->m_pPoints = NULL;
	pShape->m_cVertexColors = 0;
	pShape->m_pVertexColors = NULL;
	pShape->m_cNormals = 0;
	pShape->m_pNormals = NULL;
	pShape->m_cUVs = 0;
	pShape->m_pUVs = NULL;
	pShape->m_cReps = 0;
	pShape->m_pReps = NULL;
	pShape->m_cFaces = 0;
	pShape->m_pFaces = NULL;
    pShape->m_cFaceIndices = 0;
	pShape->m_cBones = 0;
	pShape->m_pBones = NULL;
	pShape->m_cMaxBonesPerFace   = 0;
	pShape->m_cMaxBonesPerPoint = 0;
	


	mmWorldTransform = mdpTransform.inclusiveMatrix();	// load the mesh's world transform (needed if skinning info is found)	

	if (!mdpMesh.hasFn(MFn::kMesh))
	{
		cout << "LoadMesh(): Input path was not a mesh as expected.  Aborting..." << endl;
		goto e_Exit;
	}

	stat = fnMesh.setObject(mdpMesh);
	if (!stat)
	{
		cout << "LoadMesh(): Aborting because object could not be read as mesh." << endl;
		goto e_Exit;
	}

	// check if mesh is visible
	bVisibility = true;
	bLodVisibility = true;
	bOverrideEnabled = false;
	bOverrideVisibility = true;

	do	// ONCE
	{
		MPlug mpVisibility, mpLodVisibility, mpOverrideEnabled, mpOverrideVisibility;

		mpVisibility = fnMesh.findPlug("visibility", &stat);
		if (!stat)
			break;

		stat = mpVisibility.getValue(bVisibility);
		if (!stat)
			break;

		mpLodVisibility = fnMesh.findPlug("lodVisibility", &stat);
		if (!stat)
			break;

		stat = mpLodVisibility.getValue(bLodVisibility);
		if (!stat)
			break;

		mpOverrideEnabled = fnMesh.findPlug("overrideEnabled", &stat);
		if (!stat)
			break;

		stat = mpOverrideEnabled.getValue(bOverrideEnabled);
		if (!stat)
			break;

		mpOverrideVisibility = fnMesh.findPlug("overrideVisibility", &stat);
		if (!stat)
			break;

		stat = mpOverrideVisibility.getValue(bOverrideVisibility);
		if (!stat)
			break;
	}
	while (false);

	// if mesh is not visible then skip it
	if (!(bVisibility && bLodVisibility && (!bOverrideEnabled || bOverrideVisibility)))
	{
		cout << "Skipping mesh \"" << fnMesh.name().asChar() << "\" because it is invisible..." << endl;
		goto e_Exit;
	}

	if (NULL == (pShape->m_szName = new char[1 + fnMesh.name().length()]))
	{
		hr = E_OUTOFMEMORY;
		goto e_Exit;
	}
	strcpy(pShape->m_szName, fnMesh.name().asChar());
	
	for (pc = pShape->m_szName; *pc != '\0'; pc++)	
	{	// replace '|' and ' ' characters by '_' in the maya partial pathname
		if (*pc == ' ' || *pc == '|')
			*pc = '_';
	}
	
	hr = g_Arrays.Add(pShape->m_szName);
	if (FAILED(hr))
		goto e_Exit;

	cout << "\t\tLoading mesh \"" << fnMesh.name().asChar() << "\"..." << endl;

	// get instance number of mesh (if mesh is uninstanced then default is 0)
	iInstance = 0;
	if (mdpMesh.isInstanced()) 
		iInstance = mdpMesh.instanceNumber();

#if 0
	{
		MItMeshVertex mitmv(fnMesh.object(), &stat);
		if (!stat)
			cout << "LoadMesh(): Could not iterate over vertices.\n";

		for (; !mitmv.isDone(); mitmv.next())
		{
			MItDependencyGraph mitdg(fnMesh.object(), MFn::kJoint, 
				MItDependencyGraph::kUpstream, MItDependencyGraph::kDepthFirst , 
				MItDependencyGraph::kNodeLevel, &stat);
			if (!stat)
				continue;
			for (; !mitdg.isDone(); mitdg.next())
			{
				MFnDagNode fnDag(mitdg.thisNode(), &stat);
				if (!stat);
//					cout << "Error: fnDag::fnDag().\n";

//				cout << fnDag.name() << "\n";
			}
		}
		cout << "count = " << mitmv.count() << ".\n";
	}
#endif


	pShape->m_cPoints = (DWORD)fnMesh.numVertices();	// point count
	if (pShape->m_cPoints > 0)
	{
		if (NULL == (pShape->m_pPoints = new FLOAT3[pShape->m_cPoints]))
		{
			cout << "LoadMesh(): Could not allocate memory for vertices.\n";
			hr = E_OUTOFMEMORY;
			goto e_Exit;
		}
	}
	else
	{
		cout << "LoadMesh(): Aborting because no vertices were found." << endl;
		goto e_Exit;
	}

	pShape->m_cNormals = (DWORD)fnMesh.numNormals();		// normal count
	if (pShape->m_cNormals > 0)
	{
		if (NULL == (pShape->m_pNormals = new FLOAT3[pShape->m_cNormals]) ||
            NULL == (piNormalRemap = new DWORD[pShape->m_cNormals]))
		{
			cout << "LoadMesh(): Could not allocate memory for normals.\n";
			hr = E_OUTOFMEMORY;
			goto e_Exit;
		}
	}

	pShape->m_cUVs = (DWORD)fnMesh.numUVs();			// uv count
	if (pShape->m_cUVs > 0)
	{
		if (NULL == (pShape->m_pUVs = new FLOAT2[pShape->m_cUVs]) ||
            NULL == (piUVRemap = new DWORD[pShape->m_cUVs]))
		{
			cout << "LoadMesh(): Could not allocate memory for UVs.\n";
			hr = E_OUTOFMEMORY;
			goto e_Exit;
		}
	}


	// MATERIALS

	stat = fnMesh.getConnectedShaders(iInstance, maoShaders, maiPolyShaderMap);
	if (!stat)
	{
		cout << "LoadMesh(): Aborting because shaders could not be found." << endl;
		goto e_Exit;
	}

	pShape->m_cGroups = maoShaders.length();

	if (!g_bExportMaterials)
		pShape->m_cGroups = 0;

	if (0 == pShape->m_cGroups)
		goto e_ExitMaterials;
	
	pShape->m_pGroups = new SGroup[pShape->m_cGroups];
	if (pShape->m_pGroups == NULL) 
	{
		hr = E_OUTOFMEMORY;
		cout << "LoadMesh(): Could not allocate memory for materials." << endl;
		goto e_Exit;
	}

	for (iGroup = 0; iGroup < pShape->m_cGroups; iGroup++)
	{
		MPlug mpShader = MFnDependencyNode(maoShaders[iGroup]).findPlug("surfaceShader", &stat);
		if (!stat || mpShader.isNull())
		{
			cout << "WARNING: Using default material because shader was not a surfaceShader." << endl;
			continue;
		}
		else
		{
			MPlugArray aPlugs;
			MFnDependencyNode fnShaderNode;

			stat = fnShaderNode.setObject(mpShader.node());
			if (stat)	// on success
			{
				// get material name
				pShape->m_pGroups[iGroup].m_szName = new char[1 + fnShaderNode.name().length()];
				if (NULL == pShape->m_pGroups[iGroup].m_szName) 
				{
					hr = E_OUTOFMEMORY;
					cout << "LoadMesh(): Could not allocate material name." << endl;
					goto e_Exit;
				}

				strcpy(pShape->m_pGroups[iGroup].m_szName, fnShaderNode.name().asChar());

				for (pc = pShape->m_pGroups[iGroup].m_szName; *pc != '\0'; pc++)	
				{	// replace '|' and ' ' characters by '_' in the maya partial pathname
					if (*pc == ' ' || *pc == '|')
						*pc = '_';
				}

				g_Arrays.Add(pShape->m_pGroups[iGroup].m_szName);
			}


			mpShader.connectedTo(aPlugs, true, false, &stat);
			if (!stat || aPlugs.length() != 1)
			{
				cout << "WARNING: Using default material because there was an error getting the shader." << endl;
				continue;
			}
			
			// default values
			MColor mcDiffuse(pShape->m_pGroups[iGroup].m_f3Diffuse[0], pShape->m_pGroups[iGroup].m_f3Diffuse[1], pShape->m_pGroups[iGroup].m_f3Diffuse[2]);
			MColor mcSpecular(pShape->m_pGroups[iGroup].m_f3Specular[0], pShape->m_pGroups[iGroup].m_f3Specular[1], pShape->m_pGroups[iGroup].m_f3Specular[2]);
			MColor mcEmissive(pShape->m_pGroups[iGroup].m_f3Emissive[0], pShape->m_pGroups[iGroup].m_f3Emissive[1], pShape->m_pGroups[iGroup].m_f3Emissive[2]);
			MColor mcTransparency(pShape->m_pGroups[iGroup].m_fTransparency, pShape->m_pGroups[iGroup].m_fTransparency, pShape->m_pGroups[iGroup].m_fTransparency);
			float fDiffuseCoeff = 1.0f;

			switch (aPlugs[0].node().apiType())
			{ 
				case MFn::kLambert:
				{			
					MFnLambertShader fnShader(aPlugs[0].node(), &stat);
					if (!stat)
						break;

					fDiffuseCoeff = fnShader.diffuseCoeff();

					mcDiffuse  = fnShader.color() * fDiffuseCoeff;
					mcEmissive = fnShader.incandescence() * fnShader.glowIntensity();
					mcSpecular = mcDiffuse;	// set specular color to the diffuse color
			
					mcTransparency = fnShader.transparency();

					pShape->m_pGroups[iGroup].m_fShininess = 0.0f;

					break;
				}
				case MFn::kBlinn:
				{
					MFnBlinnShader fnShader(aPlugs[0].node(), &stat);
					if (!stat)
						break;

					fDiffuseCoeff = fnShader.diffuseCoeff();

					mcDiffuse  = fnShader.color() * fDiffuseCoeff;
					mcEmissive = fnShader.incandescence() * fnShader.glowIntensity();
					mcSpecular = fnShader.specularColor();

					mcTransparency = fnShader.transparency();
					
					pShape->m_pGroups[iGroup].m_fShininess = fnShader.reflectivity();		// TODO:  Not sure about this...

					break;
				}
				case MFn::kPhong:
				{
					MFnPhongShader fnShader(aPlugs[0].node(), &stat);
					if (!stat)
						break;

					fDiffuseCoeff = fnShader.diffuseCoeff();

					mcDiffuse  = fnShader.color() * fDiffuseCoeff;
					mcEmissive = fnShader.incandescence() * fnShader.glowIntensity();
					mcSpecular = fnShader.specularColor();
			
					mcTransparency = fnShader.transparency();

					pShape->m_pGroups[iGroup].m_fShininess = fnShader.reflectivity();		// TODO:  I'm not at all sure about this...

					break;	
				}
				default:
				{
					MFnDependencyNode fnNode(aPlugs[0].node(), &stat);
					if (!stat) 
						break;

					MPlug mpDiffuse = fnNode.findPlug("outColor");
					if (stat && mpDiffuse.numElements() >= 3) 
					{
						mpDiffuse[0].getValue(mcDiffuse.r);
						mpDiffuse[1].getValue(mcDiffuse.g);
						mpDiffuse[2].getValue(mcDiffuse.b);

						mcSpecular = mcDiffuse;
					}

					MPlug mpEmissive = fnNode.findPlug("outGlowColor", &stat);
					if (stat && mpEmissive.numElements() >= 3) 
					{
						mpEmissive[0].getValue(mcDiffuse.r);
						mpEmissive[1].getValue(mcDiffuse.g);
						mpEmissive[2].getValue(mcDiffuse.b);
					}

					MPlug mpTransparency = fnNode.findPlug("outTransparency", &stat);
					if (stat && mpTransparency.numElements() >= 3) 
					{
						mpTransparency[0].getValue(mcDiffuse.r);
						mpTransparency[1].getValue(mcDiffuse.g);
						mpTransparency[2].getValue(mcDiffuse.b);
					}

					break;
				}
			}
			
			pShape->m_pGroups[iGroup].m_f3Diffuse[0] = mcDiffuse.r;
			pShape->m_pGroups[iGroup].m_f3Diffuse[1] = mcDiffuse.g;
			pShape->m_pGroups[iGroup].m_f3Diffuse[2] = mcDiffuse.b;

			pShape->m_pGroups[iGroup].m_f3Specular[0] = mcSpecular.r;
			pShape->m_pGroups[iGroup].m_f3Specular[1] = mcSpecular.g;
			pShape->m_pGroups[iGroup].m_f3Specular[2] = mcSpecular.b;

			pShape->m_pGroups[iGroup].m_f3Emissive[0] = mcEmissive.r;
			pShape->m_pGroups[iGroup].m_f3Emissive[1] = mcEmissive.g;
			pShape->m_pGroups[iGroup].m_f3Emissive[2] = mcEmissive.b;

			// TODO:  not sure if this is the best way to get transparency
			// take root mean square of the components of mcolTransparency
			mcTransparency *= mcTransparency;
			pShape->m_pGroups[iGroup].m_fTransparency = (float)sqrt((mcTransparency.r + mcTransparency.g + mcTransparency.b) / 3);

			// find texture file info if available
			MPlug mpColor = MFnDependencyNode(aPlugs[0].node()).findPlug("color", &stat);
			if (!stat)
				continue;

			MItDependencyGraph dgIt(mpColor, MFn::kFileTexture, MItDependencyGraph::kUpstream, MItDependencyGraph::kBreadthFirst, MItDependencyGraph::kNodeLevel, &stat);
			if (!stat)
				continue;
			
			dgIt.disablePruningOnFilter();

			if (dgIt.isDone())
				continue;	// no texture file node was found, so just continue.
			  
			// get the texture file name
			MString msTextureFile;
			MFnDependencyNode(dgIt.thisNode()).findPlug("fileTextureName").getValue(msTextureFile);

			pShape->m_pGroups[iGroup].m_szTextureFile = NULL;

			if (g_bRelativeTexFile)
			{	// use a trick to get the relative file name without too much hassle
				MFileObject mFile;

				mFile.setFullName(msTextureFile);

				pShape->m_pGroups[iGroup].m_szTextureFile = new char[mFile.name().length() + 1];
				if (pShape->m_pGroups[iGroup].m_szTextureFile == NULL)
				{
					hr = E_OUTOFMEMORY;
					cout << "LoadMesh(): Not enough memory for texture file name." << endl;
					goto e_Exit;
				}

				strcpy(pShape->m_pGroups[iGroup].m_szTextureFile, mFile.name().asChar());
			}
			else
			{	// directly use the texture file name
				pShape->m_pGroups[iGroup].m_szTextureFile = new char[msTextureFile.length() + 1];
				if (pShape->m_pGroups[iGroup].m_szTextureFile == NULL)
				{
					hr = E_OUTOFMEMORY;
					cout << "LoadMesh(): Not enough memory for texture file name." << endl;
					goto e_Exit;
				}

				strcpy(pShape->m_pGroups[iGroup].m_szTextureFile, msTextureFile.asChar());
			}

			g_Arrays.Add(pShape->m_pGroups[iGroup].m_szTextureFile);

			// TODO: in case of texture, use diffuse coeff as diffuse color? this is just a guess
			pShape->m_pGroups[iGroup].m_f3Diffuse[0] = fDiffuseCoeff;
			pShape->m_pGroups[iGroup].m_f3Diffuse[1] = fDiffuseCoeff;
			pShape->m_pGroups[iGroup].m_f3Diffuse[2] = fDiffuseCoeff;
		}
	}

e_ExitMaterials:

	if (0 == pShape->m_cGroups && 0 < pShape->m_cUVs)
	{	// don't bother to have UVs when there are no materials anyway
		delete[] pShape->m_pUVs;
		pShape->m_pUVs = NULL;
		pShape->m_cUVs = 0;

		// create a default material
		pShape->m_cGroups = 1;
		if (NULL == (pShape->m_pGroups = new SGroup[pShape->m_cGroups]))
		{
			hr = E_OUTOFMEMORY;
			cout << "LoadMesh(): Could not allocate memory for materials." << endl;
			goto e_Exit;
		}
	}



	// SKINNING

	acBonesPerPoint = new DWORD[pShape->m_cPoints];
	if (acBonesPerPoint == NULL)
	{
		hr = E_OUTOFMEMORY;
		cout << "LoadMesh(): Could not allocate memory for bones-per-point counts." << endl;
		goto e_Exit;
	}

	memset(acBonesPerPoint, 0, pShape->m_cPoints * sizeof(DWORD));	// zero out the bone counts

	pShape->m_pBones = new SBone[cBonesMax];	// NOTE: cBonesMax is initialized to 10
	if (pShape->m_pBones == NULL)
	{
		hr = E_OUTOFMEMORY;
		cout << "LoadMesh(): Could not allocate memory for bones." << endl;
		goto e_Exit;
	}

	aabBonePointTable = new bool*[cBonesMax];
	if (aabBonePointTable == NULL)
	{
		hr = E_OUTOFMEMORY;
		cout << "LoadMesh(): Could not allocate memory for point/bone table." << endl;
		goto e_Exit;
	}

	memset(aabBonePointTable, 0, cBonesMax * sizeof(bool*));	// NULLify the array



	if (!g_bExportSkin)
		goto e_ExitSkinning;

	// SMOOTH SKINNING

	stat = itDepNodes.reset(MFn::kSkinClusterFilter);
	if (!stat)
	{
		cout << "WARNING: Could not search skin clusters." << endl;
		goto e_ExitSmoothSkinning;
	}

	for (bSmoothSkinning = false; !itDepNodes.isDone(); itDepNodes.next()) 
	{
		MDagPathArray dagBonePaths;

		stat = fnSkin.setObject(itDepNodes.item());
		if (!stat)
		{
			cout << "WARNING: Ignoring this skin cluster because couldn't read the object." << endl;
			continue;
		}

		fnSkin.indexForOutputShape(fnMesh.object(), &stat);
		if (!stat)
			continue;	// our mesh is not an output shape for this skin cluster so skip this cluster

		if (!bSmoothSkinning)
		{
			cout << "\t\t\tSmooth skinning:";
			bSmoothSkinning = true;
		}

		// bone DAG paths
		fnSkin.influenceObjects(dagBonePaths, &stat);
		if (!stat)
		{
			cout << "WARNING: Ignoring this skin cluster because could not get influence objects." << endl;
			continue;
		}

		for (iBone = 0; iBone < dagBonePaths.length(); iBone++)
		{
			MFloatArray Weights;
			MSelectionList SelectionList;
			MDagPath dagPath;

			cout << " " << dagBonePaths[iBone].partialPathName().asChar();

			// load affected points & weights
			stat = fnSkin.getPointsAffectedByInfluence(dagBonePaths[iBone], SelectionList, Weights);
			if (!stat)
			{
				cout << "WARNING: Ignoring bone because skin weights could not be obtained." << endl;
				continue;
			}

			cWeights = Weights.length();
			if (0 == cWeights)
				continue;	// no influences so ignore this bone

			if (1 != SelectionList.length())
			{	// unexpected selection list size
				cout << "WARNING: Ignoring bone because length of selection list is not 1." << endl;
				continue;
			}

			stat = SelectionList.getDagPath(0, dagPath, moComponents);
			if (!stat)
			{
				cout << "WARNING: Ignoring bone because could not read DAG path from selection list." << endl;
				continue;
			}

			if (!(dagPath == mdpMesh))
			{	// bone does not influence this mesh
				cout << "WARNING: Ignoring bone because unexpected affected mesh encountered." << endl;
				continue;
			}


			if (!dagBonePaths[iBone].hasFn(MFn::kTransform))
			{
				cout << "WARNING: Ignoring bone because it is not a transform." << endl;
				continue;
			}
			
			stat = fnBone.setObject(dagBonePaths[iBone]);
			if (!stat)
			{
				cout << "WARNING: Ignoring bone because could not create function set." << endl;
				continue;
			}

			// BIND POSE
			moWorldBindMatrix = MObject::kNullObj;
			moLocalBindMatrix = MObject::kNullObj;

			mpToBindPose = fnBone.findPlug("bindPose", &stat);
			if (stat) // success
			{
				stat = mpToBindPose.getValue(moWorldBindMatrix);
				if (!stat)
				{
					cout << "WARNING: Ignoring bone. (Could not get 'bindPose' plug value.)" << endl;
					continue;
				}
			}
			else // failure
			{
				mpToBindPose = fnBone.findPlug("message", &stat);
				if (!stat)
				{
					cout << "WARNING: Ignoring bone. (Could not find 'message' plug.)" << endl;
					continue;
				}
			}

			// attempt to read local transform (preferable), and possibly world transform (if it wasn't read before)
			do
			{
				MPlug mpDagPose;
				MFnDependencyNode fnDagPose;
				MPlugArray mapConnections;
				MObject moAttribWM, moAttribXM;

				mpToBindPose.connectedTo(mapConnections, false, true, &stat);
				if (!stat)
					break;

				if (mapConnections.length() == 0)
				{
					stat = MS::kFailure;
					break;
				}

				mpDagPose = mapConnections[0];	// TODO: search through all connections instead of simply picking the first one.

				stat = fnDagPose.setObject(mpDagPose.node());
				if (!stat)
					break;

				// world bind pose matrix (overwrites moWorldBindMatrix if read previously)
				while (moWorldBindMatrix.isNull())
				{
					moAttribWM = fnDagPose.attribute("worldMatrix", &stat);
					if (!stat)
						break;

					MPlug mpWorldMatrix(mpDagPose.node(), moAttribWM);

					stat = mpWorldMatrix.selectAncestorLogicalIndex(mpDagPose.logicalIndex(), moAttribWM);
					if (!stat)
						break;

					stat = mpWorldMatrix.getValue(moWorldBindMatrix);
					if (!stat)
						break;
				}

				// local bind pose matrix
				while (moLocalBindMatrix.isNull())
				{
					moAttribXM = fnDagPose.attribute("xformMatrix", &stat);
					if (!stat)
						break;

					MPlug mpXformMatrix(mpDagPose.node(), moAttribXM);

					stat = mpXformMatrix.selectAncestorLogicalIndex(mpDagPose.logicalIndex(), moAttribXM);
					if (!stat)
						break;

					stat = mpXformMatrix.getValue(moLocalBindMatrix);
					if (!stat)
						break;
				}

				break;
			} while (false);

			if (moWorldBindMatrix.isNull() && moLocalBindMatrix.isNull())
			{
				cout << "WARNING: Ignoring bone. (Neither local nor world bind matrices could be obtained.)" << endl;
				continue;
			}

			
			// get bone name
			delete[] szBone;
			szBone = NULL;
			szBone = new char[1 + dagBonePaths[iBone].partialPathName().length()];
			if (NULL == szBone) 
			{
				hr = E_OUTOFMEMORY;
				cout << "LoadMesh(): Could not allocate bone name." << endl;
				goto e_Exit;
			}

			strcpy(szBone, dagBonePaths[iBone].partialPathName().asChar());

			for (pc = szBone; *pc != '\0'; pc++)	
			{	// replace '|' and ' ' characters by '_' in the maya partial pathname
				if (*pc == ' ' || *pc == '|')
					*pc = '_';
			}


			iBoneIdx = pShape->m_cBones;

			// add bone
			if (iBoneIdx == cBonesMax)
			{	// double array size
				SBone* aBones = NULL;
				bool** aabBPT = NULL;

				cBonesMax *= 2;

				aBones = new SBone[cBonesMax];
				if (NULL == aBones)
				{
					cout << "LoadMesh(): Could not allocate bones." << endl;
					goto e_Exit;
				}

				memcpy(aBones, pShape->m_pBones, pShape->m_cBones * sizeof(SBone));

				aabBPT = new bool*[cBonesMax];
				if (NULL == aabBPT)
				{
					cout << "LoadMesh(): Could not allocate bones." << endl;
					goto e_Exit;
				}

				memset(aabBPT, 0, cBonesMax * sizeof(bool*));	// NULLify the table
				memcpy(aabBPT, aabBonePointTable, pShape->m_cBones * sizeof(bool*));

				delete[] pShape->m_pBones;
				delete[] aabBonePointTable;

				pShape->m_pBones = aBones;
				aabBonePointTable = aabBPT;
			}

			if (iBoneIdx == pShape->m_cBones)
			{
				aabBonePointTable[iBoneIdx] = new bool[pShape->m_cPoints];
				if (NULL == aabBonePointTable)
				{
					cout << "LoadMesh(): Could not allocate new column for bone point table." << endl;
					goto e_Exit;
				}

				memset(aabBonePointTable[iBoneIdx], 0, pShape->m_cPoints * sizeof(bool));	// falsify entries

				pShape->m_pBones[iBoneIdx].m_szName = szBone;
				g_Arrays.Add(pShape->m_pBones[iBoneIdx].m_szName);
				szBone = NULL;

				maoBones.append(MObject::kNullObj);
				maoWorldBindMatrices.append(MObject::kNullObj);
				maoLocalBindMatrices.append(MObject::kNullObj);

				pShape->m_cBones++;
			}

			// save object & bind position
			maoBones[iBoneIdx] = fnBone.object();
			maoWorldBindMatrices[iBoneIdx] = moWorldBindMatrix;
			maoLocalBindMatrices[iBoneIdx] = moLocalBindMatrix;


			// get the points & weights
			afWeights = NULL;
			afWeights = new float[pShape->m_pBones[iBoneIdx].m_cWeights + cWeights];
			if (NULL == afWeights)
			{
				hr = E_OUTOFMEMORY;
				cout << "LoadMesh(): Could not allocate memory for bone weights." << endl;
				goto e_Exit;
			}
			memcpy (afWeights, pShape->m_pBones[iBoneIdx].m_pfWeights, pShape->m_pBones[iBoneIdx].m_cWeights * sizeof(float));
			delete[] pShape->m_pBones[iBoneIdx].m_pfWeights;
			pShape->m_pBones[iBoneIdx].m_pfWeights  = afWeights;
			afWeights = NULL;

			aiPoints = NULL;
			aiPoints = new DWORD[pShape->m_pBones[iBoneIdx].m_cWeights + cWeights];
			if (NULL == aiPoints)
			{
				hr = E_OUTOFMEMORY;
				cout << "LoadMesh(): Could not allocate memory for bone influenced points." << endl;
				goto e_Exit;
			}
			memcpy (aiPoints, pShape->m_pBones[iBoneIdx].m_piPoints, pShape->m_pBones[iBoneIdx].m_cWeights * sizeof(DWORD));
			delete[] pShape->m_pBones[iBoneIdx].m_piPoints;
			pShape->m_pBones[iBoneIdx].m_piPoints = aiPoints;
			aiPoints = NULL;

			MItGeometry itPoints(dagPath, moComponents);
			if ((DWORD)itPoints.count() != cWeights)
			{
				hr = E_FAIL;
				cout << "WARNING: Ignoring bone because number of points differed from number of weights." << endl;
				goto e_Exit;
			}

			for (iWeight = 0; !itPoints.isDone(); itPoints.next(), iWeight++)
			{
				DWORD iIndex = itPoints.index();

				if (iIndex >= pShape->m_cPoints)
				{
					cout << "WARNING: Bone affects invalid index - ignoring..." << endl;
					continue;
				}

				if (aabBonePointTable[iBoneIdx][iIndex])
				{
					cout << "WARNING: Ignoring the effect of this bone on this point because the point was previously affected by this bone before." << endl;
					continue;
				}

				pShape->m_pBones[iBoneIdx].m_piPoints[pShape->m_pBones[iBoneIdx].m_cWeights] = iIndex;
				pShape->m_pBones[iBoneIdx].m_pfWeights[pShape->m_pBones[iBoneIdx].m_cWeights] = Weights[iWeight];
				pShape->m_pBones[iBoneIdx].m_cWeights++;
				// NOTE: we will calculate pShape->m_pBones[iBoneIdx].m_cReps later
//				pShape->m_pBones[iBoneIdx].m_cReps += pShape->m_pReps[iIndex].m_cReps;

				acBonesPerPoint[iIndex]++;
				aabBonePointTable[iBoneIdx][iIndex] = true;

			}	// loop through weights

		}	// loop through bones

	}	// loop through skin clusters


e_ExitSmoothSkinning:

	if (bSmoothSkinning)
		cout << endl;



	// RIGID SKINNING

	stat = itDepNodes.reset(MFn::kJointCluster);
	if (!stat)
	{
		cout << "WARNING: Could not search joint clusters." << endl;
		goto e_ExitRigidSkinning;
	}

	for (bRigidSkinning = false; !itDepNodes.isDone(); itDepNodes.next()) 
	{
		MFnDagNode fnDagNode;
		DWORD iSet;
		MObject moJointAttrib, moDeformerSet;
		MFnDependencyNode fnNode;
		MFnSet fnSet;
		MPlugArray aPlugs;
		MSelectionList SetList;
		MDagPath dagPath;
		MFloatArray Weights;

		stat = fnCluster.setObject(itDepNodes.item());
		if (!stat)
		{
			cout << "WARNING: Ignoring this joint cluster because couldn't read the object." << endl;
			continue;
		}

		fnCluster.indexForOutputShape(fnMesh.object(), &stat);
		if (!stat)
			continue;	// our mesh is not an output shape for this skin cluster so skip this cluster


		// get joint
		if (!itDepNodes.item().hasFn(MFn::kDependencyNode))
		{
			cout << "WARNING: Ignoring joint cluster because object was not a dependency node." << endl;
			continue;
		}

		stat = fnNode.setObject(itDepNodes.item());
		if (!stat)
		{
			cout << "WARNING: Ignoring joint cluster because dependency node could not be read." << endl;
			continue;
		}

		moJointAttrib = fnNode.attribute("matrix", &stat);
		if (!stat)
		{
			cout << "WARNING: Ignoring joint cluster because could not get 'matrix' attribute." << endl;
			continue;
		}

		if (!MPlug(fnNode.object(), moJointAttrib).connectedTo(aPlugs, true, false, &stat) || !stat || aPlugs.length() != 1)
		{
			cout << "WARNING: Ignoring joint cluster because there was a problem getting connections to plug." << endl;
			continue;
		}

		if (aPlugs[0].node().isNull() || !aPlugs[0].node().hasFn(MFn::kTransform))
		{
			cout << "WARNING: Ignoring joint cluster because no transform attached to this cluster." << endl;
			continue;
		}

		stat = fnBone.setObject(aPlugs[0].node());
		if (!stat)
		{
			cout << "WARNING: Ignoring joint cluster because could not read joint object." << endl;
			continue;
		}

		// get deformed objects
		moDeformerSet = fnCluster.deformerSet(&stat);
		if (!stat)
		{
			cout << "WARNING: Ignoring joint cluster because could not get deformer set object." << endl;
			continue;
		}

		stat = fnSet.setObject(moDeformerSet);	//need the fn to get the members
		if (!stat)
		{
			cout << "WARNING: Ignoring joint cluster because could not read deformer set object." << endl;
			continue;
		}
		  

		stat = fnSet.getMembers(SetList, true);
		if (!stat)
		{
			cout << "WARNING: Ignoring joint cluster because could not get members from set list." << endl;
			continue;
		}


		for (iSet = 0; iSet < SetList.length(); iSet++) 
		{
			stat = SetList.getDagPath(iSet, dagPath, moComponents);
			if (stat && dagPath == mdpMesh)
				break;
		}

		if (iSet == SetList.length())
		{
			cout << "WARNING: Ignoring joint cluster because mesh DAG path was not found in set list." << endl;
			continue;
		}

		// get affected weights & points
		stat = fnCluster.getWeights(dagPath, moComponents, Weights);
		if (!stat)
		{
			cout << "WARNING: Ignoring joint cluster because could not read weights & components." << endl;
			continue;
		}

		cWeights = Weights.length();
		if (0 == cWeights)
			continue;


		// BIND POSE
		moWorldBindMatrix = MObject::kNullObj;
		moLocalBindMatrix = MObject::kNullObj;

		mpToBindPose = fnBone.findPlug("bindPose", &stat);
		if (stat) // success
		{
			stat = mpToBindPose.getValue(moWorldBindMatrix);
			if (!stat)
			{
				cout << "WARNING: Ignoring bone. (Could not get 'bindPose' plug value.)" << endl;
				continue;
			}
		}
		else // failure
		{
			mpToBindPose = fnBone.findPlug("message", &stat);
			if (!stat)
			{
				cout << "WARNING: Ignoring bone. (Could not find 'message' plug.)" << endl;
				continue;
			}
		}

		// attempt to read local transform (preferable), and possibly world transform (if it wasn't read before)
		do
		{
			MPlug mpDagPose;
			MFnDependencyNode fnDagPose;
			MPlugArray mapConnections;
			MObject moAttribWM, moAttribXM;

			mpToBindPose.connectedTo(mapConnections, false, true, &stat);
			if (!stat)
				break;

			if (mapConnections.length() == 0)
			{
				stat = MS::kFailure;
				break;
			}

			mpDagPose = mapConnections[0];	// TODO: search through all connections instead of simply picking the first one.

			stat = fnDagPose.setObject(mpDagPose.node());
			if (!stat)
				break;

			// world bind pose matrix (overwrites moWorldBindMatrix if read previously)
			while (moWorldBindMatrix.isNull())
			{
				moAttribWM = fnDagPose.attribute("worldMatrix", &stat);
				if (!stat)
					break;

				MPlug mpWorldMatrix(mpDagPose.node(), moAttribWM);

				stat = mpWorldMatrix.selectAncestorLogicalIndex(mpDagPose.logicalIndex(), moAttribWM);
				if (!stat)
					break;

				stat = mpWorldMatrix.getValue(moWorldBindMatrix);
				if (!stat)
					break;
			}

			// local bind pose matrix
			while (moLocalBindMatrix.isNull())
			{
				moAttribXM = fnDagPose.attribute("xformMatrix", &stat);
				if (!stat)
					break;

				MPlug mpXformMatrix(mpDagPose.node(), moAttribXM);

				stat = mpXformMatrix.selectAncestorLogicalIndex(mpDagPose.logicalIndex(), moAttribXM);
				if (!stat)
					break;

				stat = mpXformMatrix.getValue(moLocalBindMatrix);
				if (!stat)
					break;
			}

			break;
		} while (false);

		if (moWorldBindMatrix.isNull() && moLocalBindMatrix.isNull())
		{
			cout << "WARNING: Ignoring bone. (Neither local nor world bind matrices could be obtained.)" << endl;
			continue;
		}



		// get bone name
		delete[] szBone;
		szBone = NULL;
		szBone = new char[1 + fnBone.partialPathName().length()];
		if (NULL == szBone) 
		{
			hr = E_OUTOFMEMORY;
			cout << "LoadMesh(): Could not allocate bone name." << endl;
			goto e_Exit;
		}

		strcpy(szBone, fnBone.partialPathName().asChar());

		for (pc = szBone; *pc != '\0'; pc++)	
		{	// replace '|' and ' ' characters by '_' in the maya partial pathname
			if (*pc == ' ' || *pc == '|')
				*pc = '_';
		}

		for (iBoneIdx = 0; iBoneIdx < pShape->m_cBones; iBoneIdx++)
		{
			if (!strcmp(pShape->m_pBones[iBoneIdx].m_szName, szBone))
				break;
		}

		// new bone
		if (iBoneIdx == cBonesMax)
		{	// double array size
			SBone* aBones;
			bool** aabBPT;

			cBonesMax *= 2;

			aBones = NULL;
			aBones = new SBone[cBonesMax];
			if (NULL == aBones)
			{
				cout << "LoadMesh(): Could not allocate bones." << endl;
				goto e_Exit;
			}

			memcpy(aBones, pShape->m_pBones, pShape->m_cBones * sizeof(SBone));

			aabBPT = NULL;
			aabBPT = new bool*[cBonesMax];
			if (NULL == aabBPT)
			{
				cout << "LoadMesh(): Could not allocate bones." << endl;
				goto e_Exit;
			}

			memset(aabBPT, 0, cBonesMax * sizeof(bool*));	// NULLify the table
			memcpy(aabBPT, aabBonePointTable, pShape->m_cBones * sizeof(bool*));

			delete[] pShape->m_pBones;
			delete[] aabBonePointTable;

			pShape->m_pBones = aBones;
			aabBonePointTable = aabBPT;
		}


		if (iBoneIdx == pShape->m_cBones)
		{	// initialize new bone
			aabBonePointTable[iBoneIdx] = new bool[pShape->m_cPoints];
			if (NULL == aabBonePointTable)
			{
				cout << "LoadMesh(): Could not allocate new column for bone point table." << endl;
				goto e_Exit;
			}

			memset(aabBonePointTable[iBoneIdx], 0, pShape->m_cPoints * sizeof(bool));	// falsify entries

			// save bone name
			pShape->m_pBones[iBoneIdx].m_szName = szBone;
			g_Arrays.Add(pShape->m_pBones[iBoneIdx].m_szName);
			szBone = NULL;	// so that we won't later delete it

			maoBones.append(MObject::kNullObj);
			maoWorldBindMatrices.append(MObject::kNullObj);
			maoLocalBindMatrices.append(MObject::kNullObj);

			pShape->m_cBones++;
		}

		// save object & bind position
		maoBones[iBoneIdx] = fnBone.object();
		maoWorldBindMatrices[iBoneIdx] = moWorldBindMatrix;
		maoLocalBindMatrices[iBoneIdx] = moLocalBindMatrix;


		// get the points & weights
		afWeights = NULL;
		afWeights = new float[pShape->m_pBones[iBoneIdx].m_cWeights + cWeights];
		if (NULL == afWeights)
		{
			hr = E_OUTOFMEMORY;
			cout << "LoadMesh(): Could not allocate memory for bone weights." << endl;
			goto e_Exit;
		}

		memcpy (afWeights, pShape->m_pBones[iBoneIdx].m_pfWeights, pShape->m_pBones[iBoneIdx].m_cWeights * sizeof(FLOAT));
		delete[] pShape->m_pBones[iBoneIdx].m_pfWeights;
		pShape->m_pBones[iBoneIdx].m_pfWeights  = afWeights;
		afWeights = NULL;

		aiPoints = NULL;
		aiPoints = new DWORD[pShape->m_pBones[iBoneIdx].m_cWeights + cWeights];
		if (NULL == aiPoints)
		{
			hr = E_OUTOFMEMORY;
			cout << "LoadMesh(): Could not allocate memory for bone influenced points." << endl;
			goto e_Exit;
		}

		memcpy (aiPoints, pShape->m_pBones[iBoneIdx].m_piPoints, pShape->m_pBones[iBoneIdx].m_cWeights * sizeof(DWORD));
		delete[] pShape->m_pBones[iBoneIdx].m_piPoints;
		pShape->m_pBones[iBoneIdx].m_piPoints = aiPoints;
		aiPoints = NULL;

		MItGeometry itPoints(dagPath, moComponents);
		if ((DWORD)itPoints.count() != cWeights)
		{
			hr = E_FAIL;
			cout << "WARNING: Ignoring bone because number of points differed from number of weights." << endl;
			goto e_Exit;
		}
		
		for (iWeight = 0; !itPoints.isDone(); itPoints.next(), iWeight++)
		{
			DWORD iIndex = itPoints.index();

			if (iIndex >= pShape->m_cPoints)
			{
				cout << "WARNING: Bone affects invalid index - ignoring..." << endl;
				continue;
			}

			if (aabBonePointTable[iBoneIdx][iIndex])
			{
				cout << "WARNING: Ignoring the effect of this bone on this point because the point was previously affected by this bone before." << endl;
				continue;
			}

			pShape->m_pBones[iBoneIdx].m_piPoints[pShape->m_pBones[iBoneIdx].m_cWeights] = iIndex;
			pShape->m_pBones[iBoneIdx].m_pfWeights[pShape->m_pBones[iBoneIdx].m_cWeights]  = Weights[iWeight];
			pShape->m_pBones[iBoneIdx].m_cWeights++;
			// NOTE: we will calculate pShape->m_pBones[iBoneIdx].m_cReps later
//			pShape->m_pBones[iBoneIdx].m_cReps += pShape->m_pReps[iIndex].m_cReps;

			acBonesPerPoint[iIndex]++;
			aabBonePointTable[iBoneIdx][iIndex] = true;
		}

		if (!bRigidSkinning)
		{
			cout << "\t\t\tRigid skinning:";
			bRigidSkinning = true;
		}
		cout << " " << fnBone.name().asChar();

		// now add parent bone
		stat = fnDagNode.setObject(fnBone.object());
		if (!stat)
		{
			hr = E_FAIL;
			cout << "LoadMesh(): Could not get bone read DAG bone object." << endl;
			goto e_Exit;
		}

		while (fnDagNode.parentCount() > 0 && !fnDagNode.parent(0).hasFn(MFn::kTransform))
		{
			stat = fnDagNode.setObject(fnDagNode.parent(0));
			if (!stat)
			{
				hr = E_FAIL;
				cout << "LoadMesh(): Could not get parent object." << endl;
				goto e_Exit;
			}
		}

		// get parent bone name
		if (fnDagNode.parentCount() == 0)
			continue;

		stat = fnDagNode.setObject(fnDagNode.parent(0));
		if (!stat)
		{
			hr = E_FAIL;
			cout << "LoadMesh(): Could not get parent object." << endl;
			goto e_Exit;
		}

		stat = fnDagNode.getPath(dagPath);
		if (!stat)
		{
			hr = E_FAIL;
			cout << "LoadMesh(): Could not get parent's path." << endl;
			goto e_Exit;
		}

		delete[] szBone;
		szBone = NULL;
		szBone = new char[1 + dagPath.partialPathName().length()];
		if (NULL == szBone) 
		{
			hr = E_OUTOFMEMORY;
			cout << "LoadMesh(): Could not allocate bone name." << endl;
			goto e_Exit;
		}

		strcpy(szBone, dagPath.partialPathName().asChar());

		for (pc = szBone; *pc != '\0'; pc++)	
		{	// replace '|' and ' ' characters by '_' in the maya partial pathname
			if (*pc == ' ' || *pc == '|')
				*pc = '_';
		}

		for (iBoneIdx = 0; iBoneIdx < pShape->m_cBones; iBoneIdx++)
		{
			if (!strcmp(pShape->m_pBones[iBoneIdx].m_szName, szBone))
				break;
		}

		// new bone
		if (iBoneIdx == cBonesMax)
		{	// double array size
			SBone* aBones;
			bool** aabBPT;

			cBonesMax *= 2;

			aBones = NULL;
			aBones = new SBone[cBonesMax];
			if (NULL == aBones)
			{
				cout << "LoadMesh(): Could not allocate bones." << endl;
				goto e_Exit;
			}

			memcpy(aBones, pShape->m_pBones, pShape->m_cBones * sizeof(SBone));

			aabBPT = NULL;
			aabBPT = new bool*[cBonesMax];
			if (NULL == aabBPT)
			{
				cout << "LoadMesh(): Could not allocate bones." << endl;
				goto e_Exit;
			}

			memset(aabBPT, 0, cBonesMax * sizeof(bool*));	// NULLify the table
			memcpy(aabBPT, aabBonePointTable, pShape->m_cBones * sizeof(bool*));

			delete[] pShape->m_pBones;
			delete[] aabBonePointTable;

			pShape->m_pBones = aBones;
			aabBonePointTable = aabBPT;
		}


		if (iBoneIdx == pShape->m_cBones)
		{	// initialize new bone
			aabBonePointTable[iBoneIdx] = new bool[pShape->m_cPoints];
			if (NULL == aabBonePointTable)
			{
				cout << "LoadMesh(): Could not allocate new column for bone point table." << endl;
				goto e_Exit;
			}

			memset(aabBonePointTable[iBoneIdx], 0, pShape->m_cPoints * sizeof(bool));	// falsify entries

			// save bone name
			pShape->m_pBones[iBoneIdx].m_szName = szBone;
			g_Arrays.Add(pShape->m_pBones[iBoneIdx].m_szName);
			szBone = NULL;	// so that we won't later delete it


			// offset matrix - it's a subtle point that we need to overwrite this each time (especially in the case of rigid skinning)
			stat = (mmWorldTransform).get(pShape->m_pBones[iBoneIdx].m_ppfOffset);
			if (!stat)
			{
				hr = E_FAIL;
				cout << "LoadMesh(): Could not get bind pose transform." << endl;
				goto e_Exit;
			}

			maoBones.append(MObject::kNullObj);
			maoWorldBindMatrices.append(MObject::kNullObj);
			maoLocalBindMatrices.append(MObject::kNullObj);

			pShape->m_cBones++;
		}


		// get the points & weights
		afWeights = NULL;
		afWeights = new float[pShape->m_pBones[iBoneIdx].m_cWeights + cWeights];
		if (NULL == afWeights)
		{
			hr = E_OUTOFMEMORY;
			cout << "LoadMesh(): Could not allocate memory for bone weights." << endl;
			goto e_Exit;
		}

		memcpy (afWeights, pShape->m_pBones[iBoneIdx].m_pfWeights, pShape->m_pBones[iBoneIdx].m_cWeights * sizeof(float));
		delete[] pShape->m_pBones[iBoneIdx].m_pfWeights;
		pShape->m_pBones[iBoneIdx].m_pfWeights  = afWeights;
		afWeights = NULL;

		aiPoints = NULL;
		aiPoints = new DWORD[pShape->m_pBones[iBoneIdx].m_cWeights + cWeights];
		if (NULL == aiPoints)
		{
			hr = E_OUTOFMEMORY;
			cout << "LoadMesh(): Could not allocate memory for bone influenced points." << endl;
			goto e_Exit;
		}

		memcpy (aiPoints, pShape->m_pBones[iBoneIdx].m_piPoints, pShape->m_pBones[iBoneIdx].m_cWeights * sizeof(DWORD));
		delete[] pShape->m_pBones[iBoneIdx].m_piPoints;
		pShape->m_pBones[iBoneIdx].m_piPoints = aiPoints;
		aiPoints = NULL;

		itPoints.reset();

		for (iWeight = 0; !itPoints.isDone(); itPoints.next(), iWeight++)
		{
			DWORD iIndex = itPoints.index();

			if (iIndex >= pShape->m_cPoints)
			{
				cout << "WARNING: Bone affects invalid index - ignoring..." << endl;
				continue;
			}

			if (aabBonePointTable[iBoneIdx][iIndex])
			{
				cout << "WARNING: Ignoring the effect of this bone on this point because the point was previously affected by this bone before." << endl;
				continue;
			}

			if (1.0f - Weights[iWeight] == 0.0f)
				continue;

			pShape->m_pBones[iBoneIdx].m_piPoints[pShape->m_pBones[iBoneIdx].m_cWeights] = iIndex;
			pShape->m_pBones[iBoneIdx].m_pfWeights[pShape->m_pBones[iBoneIdx].m_cWeights] = 1.0f - Weights[iWeight];
			pShape->m_pBones[iBoneIdx].m_cWeights++;
			// NOTE: we will calculate pShape->m_pBones[iBoneIdx].m_cReps later
//			pShape->m_pBones[iBoneIdx].m_cReps += pShape->m_pReps[iIndex].m_cReps;

			acBonesPerPoint[iIndex]++;

			aabBonePointTable[iBoneIdx][iIndex] = true;
		}

	}


e_ExitRigidSkinning:

	if (bRigidSkinning)
		cout << endl;

e_ExitSkinning:

	// GO TO BIND POSITION

	if ((DWORD)maoBones.length() != pShape->m_cBones)	// sanity check
	{
		cout << "LoadMesh(): Aborting because there was a mismatch in the length of the bone- and the bone object- arrays." << endl;
		goto e_Exit;
	}

	amtmBonePositions = new MTransformationMatrix[pShape->m_cBones];
	if (NULL == amtmBonePositions)
	{
		cout << "LoadMesh(): Could not allocate bone transformation array." << endl;
		goto e_Exit;
	}

	// turn IK off
	bIkSnap = MIkSystem::isGlobalSnap(&stat);
	if (!stat)
	{
		cout << "WARNING: Could not determine whether global IK snapping is on.\n";
		bIkSnap = false;
	}
	else if (bIkSnap) 
	{
		stat = MIkSystem::setGlobalSnap(false);
		if (!stat)
			cout << "WARNING: Could not turn global IK snapping off.\n";
	}

	bIkSolve = MIkSystem::isGlobalSolve(&stat);
	if (!stat)
	{
		cout << "WARNING: Could not determine whether global IK solving is on.\n";
		bIkSolve = false;
	}
	else if (bIkSolve) 
	{
		stat = MIkSystem::setGlobalSolve(false);
		if (!stat)
			cout << "WARNING: Could not turn global IK solving off.\n";
	}

	// create bone remap array based on descending hierarchical order
	for (iBone = 0; iBone < pShape->m_cBones; iBone++)
	{
		if (maoBones[iBone].isNull() || !maoBones[iBone].hasFn(MFn::kJoint))
			continue;

		stat = fnBone.setObject(maoBones[iBone]);
		if (!stat)
			continue;

		amtmBonePositions[iBone] = fnBone.transformation();

		if (maoLocalBindMatrices[iBone].isNull())
			continue;
		
		stat = fnMatrix.setObject(maoLocalBindMatrices[iBone]);
		if (!stat)
			continue;

		stat = fnBone.set(fnMatrix.transformation());
		if (!stat)
			continue;
	}

	for (; !itBones.isDone(); itBones.next())
	{
		MObject moBone = itBones.item();

		for (iBone = 0; iBone < pShape->m_cBones; iBone++)
		{
			if (!maoLocalBindMatrices[iBone].isNull())
				break;

			if (moBone == maoBones[iBone])
			{
				maiBoneRemap.append(iBone);
				break;
			}
		}
	}

	for (iBoneRemap = 0; iBoneRemap < maiBoneRemap.length(); iBoneRemap++)
	{
		MDagPath mdpBone;
		iBone = maiBoneRemap[iBoneRemap];

		if (maoBones[iBone].isNull() || !maoBones[iBone].hasFn(MFn::kJoint))
			continue;

		stat = fnBone.setObject(maoBones[iBone]);
		if (!stat)
			continue;

		stat = fnBone.getPath(mdpBone);
		if (!stat)
			continue;

		if (maoWorldBindMatrices[iBone].isNull())
			continue;
		
		stat = fnMatrix.setObject(maoWorldBindMatrices[iBone]);
		if (!stat)
			continue;

		stat = fnBone.set(MTransformationMatrix(fnMatrix.matrix() * mdpBone.exclusiveMatrixInverse()));
		if (!stat)
			continue;
	}


	for (iBone = 0; iBone < pShape->m_cBones; iBone++)
	{
		MMatrix mmBoneOffsetMatrix = mmWorldTransform;
		MDagPath mdpBone;
		
		do 
		{
			MDagPath mdpBone;

			if (maoBones[iBone].isNull())
				break;

			stat = MDagPath::getAPathTo(maoBones[iBone], mdpBone);
			if (!stat)
			{
				cout << "WARNING: Could not get DAG path to bone.  Skinning may look incorrect." << endl;
				break;
			}

			mmBoneOffsetMatrix = mmBoneOffsetMatrix * mdpBone.inclusiveMatrix().inverse();
		} while (false);

		mmBoneOffsetMatrix.get(pShape->m_pBones[iBone].m_ppfOffset);
	}


	stat = fnMesh.syncObject();
	if (!stat)
	{
		cout << "LoadMesh(): Could not sync mesh object." << endl;
		goto e_ExitGeometry;
	}

	stat = fnMesh.syncObject();
	if (!stat)
	{
		cout << "LoadMesh(): Could not sync mesh object." << endl;
		goto e_ExitGeometry;
	}
	
	// POINTS
	stat = fnMesh.getPoints(mfpaPoints);
	if (!stat)
	{
		cout << "WARNING: Aborting because vertices could not be read." << endl;
		goto e_ExitGeometry;
	}

	if (mfpaPoints.length() != pShape->m_cPoints)
	{
		cout << "WARNING: Aborting because point array length(" << mfpaPoints.length() << ") did not match number of points(" << pShape->m_cPoints << ")." << endl;
		goto e_ExitGeometry;
	}

	for (iVert = 0; iVert < pShape->m_cPoints; iVert++) 
	{
		pShape->m_pPoints[iVert][0] = mfpaPoints[iVert][0];
		pShape->m_pPoints[iVert][1] = mfpaPoints[iVert][1];
		pShape->m_pPoints[iVert][2] = mfpaPoints[iVert][2];
	}


	// NORMALS
	if (pShape->m_cNormals == 0)
		goto e_ExitNormals;

	stat = fnMesh.getNormals(mfvaNormals);
	if (!stat)
	{
		cout << "LoadMesh(): Could not be read normals." << endl;
		goto e_ExitGeometry;
	}

	if (mfvaNormals.length() != pShape->m_cNormals)
	{
		cout << "LoadMesh: Normal array length did not match number of normals." << endl;
		goto e_ExitGeometry;
	}

	for (iNorm = 0; iNorm < pShape->m_cNormals; iNorm++)	
	{
		pShape->m_pNormals[iNorm][0] = mfvaNormals[iNorm][0];
		pShape->m_pNormals[iNorm][1] = mfvaNormals[iNorm][1];
		pShape->m_pNormals[iNorm][2] = mfvaNormals[iNorm][2];
	}

    // remove duplicate normals
    if (FAILED(hr = ReduceNormals(pShape, piNormalRemap)))
        goto e_Exit;

e_ExitNormals:


	// UVs
	if (pShape->m_cUVs == 0)
		goto e_ExitUVs;

	stat = fnMesh.getUVs(mfaUs, mfaVs);
	if (!stat)
	{
		cout << "WARNING: Aborting because UVs could not be read." << endl;
		goto e_ExitGeometry;
	}

	if (mfaUs.length() != pShape->m_cUVs || mfaVs.length() != pShape->m_cUVs)	// sanity check
	{
		cout << "WARNING: Aborting because UV array length did not match number of UVs." << endl;
		goto e_ExitGeometry;
	}

	for (iUV = 0; iUV < pShape->m_cUVs; iUV++)	
	{
		pShape->m_pUVs[iUV][0] = mfaUs[iUV];
		pShape->m_pUVs[iUV][1] = mfaVs[iUV];
	}

    // remove duplicate UVs
    if (FAILED(hr = ReduceUVs(pShape, piUVRemap)))
        goto e_Exit;

e_ExitUVs:

	// FACES & REPS
	// initialize reps
	pShape->m_cReps = pShape->m_cPoints;
	cRepsMax = pShape->m_cReps;		// NOTE: cRepsMax > 0 (because earlier we returned if m_cPoints == 0)

	if (NULL == (pShape->m_pReps = new SRep[cRepsMax]))
	{
		cout << "LoadMesh(): Could not allocate memory for point reps.\n";
		hr = E_OUTOFMEMORY;
		goto e_ExitGeometry;
	}
	
    ZeroMemory(pShape->m_pReps, pShape->m_cReps * sizeof(SRep));

	// load faces
	pShape->m_cFaces = (DWORD)fnMesh.numPolygons();

	if (0 == pShape->m_cFaces)
	{
		cout << "WARNING: Aborting because mesh contains no faces." << endl;
		goto e_ExitGeometry;
	}

	if (pShape->m_cGroups > 1 && maiPolyShaderMap.length() != pShape->m_cFaces)
	{
		cout << "WARNING: Aborting because polygon/shader map length is not equal to number of faces." << endl;
		goto e_ExitGeometry;
	}

	if (NULL == (pShape->m_pFaces = new SFace[pShape->m_cFaces]))
	{
		cout << "LoadMesh(): Could not allocate memory for faces.\n";
		hr = E_OUTOFMEMORY;
		goto e_ExitGeometry;
	}

	pShape->m_cFaceIndices = 0;

	stat = itPoly.reset(fnMesh.object());
	if (!stat)
	{
		cout << "LoadPolyMesh: Aborting because polygon iterator could not be initialized.\n";
		goto e_ExitGeometry;
	}

	for (iFace = 0; !itPoly.isDone(); iFace++, itPoly.next())
	{
		if (1 == pShape->m_cGroups)
			pShape->m_pFaces[iFace].m_iGroup = 0;
		else
			pShape->m_pFaces[iFace].m_iGroup = maiPolyShaderMap[iFace];

		if (itPoly.polygonVertexCount() == 0)
			goto e_ExitGeometry;

		pShape->m_pFaces[iFace].m_cIndices = itPoly.polygonVertexCount();
		if (NULL == (pShape->m_pFaces[iFace].m_pIndices = new DWORD[pShape->m_pFaces[iFace].m_cIndices]))
		{
			cout << "LoadMesh(): Could not allocate memory for face " << iFace << "'s indices.\n";
			hr = E_OUTOFMEMORY;
			goto e_ExitGeometry;
		}

		for (iIndex = 0; iIndex < pShape->m_pFaces[iFace].m_cIndices; iIndex++)
		{
			int iUV;
			bool bFound = false;

			iVert = itPoly.vertexIndex(iIndex, &stat);
			if (!stat)
			{
				cout << "LoadMesh(): Could not load face " << iFace << "'s vertex number " << iVert << endl;
				hr = E_FAIL;
				goto e_ExitGeometry;
			}

            iNorm = 0;
			if (0 < pShape->m_cNormals)
			{
				iNorm = itPoly.normalIndex(iIndex, &stat);
				if (!stat)
				{
					iNorm = 0;
					cout << "WARNING: Using first normal because there was an error in retrieving it." << endl;
				}

                iNorm = piNormalRemap[iNorm];
			}


            iUV = 0;
			if (0 < pShape->m_cUVs)
			{
				if (!itPoly.getUVIndex(iIndex, iUV))
				{
					iUV = 0;
					cout << "WARNING: Using first UV because there was an error in retrieving it." << endl;
				}

                iUV = piUVRemap[iUV];
			}


			if (pShape->m_pReps[iVert].m_cReps == 0) 	
			{	// first time through this rep
				pShape->m_pReps[iVert].m_iUV    = (DWORD)iUV;	
				pShape->m_pReps[iVert].m_iNorm  = iNorm;
				pShape->m_pReps[iVert].m_cReps  = 1;
		        pShape->m_pReps[iVert].m_iNext  = iVert;
		        pShape->m_pReps[iVert].m_iFirst = iVert;

				// update face indices
				pShape->m_pFaces[iFace].m_pIndices[iIndex] = iVert;	
			}
			else
			{
				DWORD iPrevRep;

				iRep = iVert;
				do 
				{
					if ((DWORD)iUV == pShape->m_pReps[iRep].m_iUV && iNorm == pShape->m_pReps[iRep].m_iNorm)
					{
						bFound = true;
						// update face indices
						pShape->m_pFaces[iFace].m_pIndices[iIndex] = iRep;	
					}

					iPrevRep = iRep;
					iRep     = pShape->m_pReps[iRep].m_iNext;
				} while (!bFound && iRep != pShape->m_pReps[iRep].m_iFirst);


				if (!bFound) 
				{	// NOTE: iRep == iVert == pShape->m_pReps[iRep].m_iFirst  
					//       (see the condition in the do...while loop above)

					// append new rep 
					if (pShape->m_cReps >= cRepsMax) 
					{	// double array size
						SRep* rgNewReps = NULL;

						cRepsMax += cRepsMax;
						
						if (NULL == (rgNewReps = new SRep[cRepsMax]))
						{
							cout << "LoadMesh(): Could not allocate memory for new rep array.\n";
							hr = E_OUTOFMEMORY;
							goto e_ExitGeometry;
						}

						memcpy(rgNewReps, pShape->m_pReps, pShape->m_cReps * sizeof(SRep));

						delete[] pShape->m_pReps;
						pShape->m_pReps = rgNewReps;
					}

					// create a new rep at the end of the array
					pShape->m_pReps[pShape->m_cReps].m_iUV = iUV;
					pShape->m_pReps[pShape->m_cReps].m_iNorm = iNorm;
					pShape->m_pReps[pShape->m_cReps].m_iFirst = iRep;	
					pShape->m_pReps[pShape->m_cReps].m_iNext = iRep;		// link the new rep to the first rep

					pShape->m_pReps[iPrevRep].m_iNext = pShape->m_cReps;	// link the last rep to the new rep

					pShape->m_pReps[iRep].m_cReps++;	// increment the rep count at the first rep

					pShape->m_pFaces[iFace].m_pIndices[iIndex] = pShape->m_cReps;	// update face indices

					pShape->m_cReps++;	// increment the total rep count
				} // if !bFound
			}

		} // for loop through indices


		pShape->m_cFaceIndices += pShape->m_pFaces[iFace].m_cIndices;

	} // for loop through faces

	if ((DWORD)fnMesh.numPolygons() != iFace)
	{
		cout << "WARNING: Aborting because of mismatch in face count." << endl;
		goto e_ExitGeometry;
	}


e_ExitGeometry:

	for (iBone = 0; iBone < pShape->m_cBones; iBone++)
	{
		if (maoBones[iBone].isNull() || !maoBones[iBone].hasFn(MFn::kJoint))
			continue;

		stat = fnBone.setObject(maoBones[iBone]);
		if (!stat)
			continue;

		stat = fnBone.set(amtmBonePositions[iBone]);
		if (!stat)
			continue;
	}

	if (bIkSnap)
	{
		stat = MIkSystem::setGlobalSnap(true);
		if (!stat)
			cout << "WARNING: Could not turn global IK snapping back on.\n";
	}

	if (bIkSolve)
	{
		stat = MIkSystem::setGlobalSolve(true);
		if (!stat)
			cout << "WARNING: Could not turn global IK solving back on.\n";
	}

	if (FAILED(hr))
	{
		cout << "LoadMesh(): Could not load geometry." << endl;
		goto e_Exit;
	}


#ifdef KAYA
#if 0
    if (0 < pShape->m_cBones)
    {
        if (FAILED(hr = InterpolateMissingSkinWeights(pShape)))
            goto e_Exit;
    }
#else
	// if there are any non-skinned vertices, then skin them with the weights of the closest skinned vertex
	if (pShape->m_cBones > 0)
	{
		DWORD i, j, k;
		DWORD iBad, iHorrible, iNearestGood;
		PDWORD rgcNeighbors = NULL;
		PDWORD* rgrgiNeighbors = NULL;
		PDWORD rgiIndices = NULL;
		PFLOAT rgfWeights = NULL;
		PFLOAT rgfVertWtSums = NULL;
		PFLOAT rgfBoneVertWts = NULL;
		FLOAT fDelta, fDistSq, fLeastDistSq;

		if (NULL == (rgcNeighbors = new DWORD[pShape->m_cPoints]) ||
			NULL == (rgrgiNeighbors = new PDWORD[pShape->m_cPoints]) ||
			NULL == (rgiIndices = new DWORD[pShape->m_cPoints]) ||
			NULL == (rgfWeights = new FLOAT[pShape->m_cPoints]) ||
			NULL == (rgfBoneVertWts = new FLOAT[pShape->m_cPoints * pShape->m_cBones]) ||
			NULL == (rgfVertWtSums = new FLOAT[pShape->m_cPoints]))
		{
			hr = E_OUTOFMEMORY;
			goto e_ExitKAYA;
		}
		
		memset(rgcNeighbors, 0, pShape->m_cPoints * sizeof(DWORD));
		memset(rgrgiNeighbors, 0, pShape->m_cPoints * sizeof(PDWORD));
		memset(rgfBoneVertWts, 0, pShape->m_cPoints * pShape->m_cBones * sizeof(FLOAT));
		memset(rgfVertWtSums, 0, pShape->m_cPoints * sizeof(FLOAT));

		// compute the table of bone/vertex weights as well as the vertex weight sums
		for (i = 0; i < pShape->m_cBones; i++)
		{
			for (j = 0; j < pShape->m_pBones[i].m_cWeights; j++)
			{
				rgfBoneVertWts[i * pShape->m_cPoints + pShape->m_pBones[i].m_piPoints[j]] += pShape->m_pBones[i].m_pfWeights[j];
				rgfVertWtSums[pShape->m_pBones[i].m_piPoints[j]] += pShape->m_pBones[i].m_pfWeights[j];
			}
		}

		for (i = 0; i < pShape->m_cFaces; i++)
			for (j = 0; j < pShape->m_pFaces[i].m_cIndices; j++)
				rgcNeighbors[pShape->m_pReps[pShape->m_pFaces[i].m_pIndices[j]].m_iFirst] += 2;

		for (i = 0; i < pShape->m_cPoints; i++)
		{
			if (NULL == (rgrgiNeighbors[i] = new DWORD[rgcNeighbors[i]]))
			{
				hr = E_OUTOFMEMORY;
				goto e_ExitKAYA;
			}
		}

		// zero this out again
		memset(rgcNeighbors, 0, pShape->m_cPoints * sizeof(DWORD));

		for (i = 0; i < pShape->m_cFaces; i++)
		{
			for (j = 0; j < pShape->m_pFaces[i].m_cIndices; j++)
			{
				k = pShape->m_pReps[pShape->m_pFaces[i].m_pIndices[j]].m_iFirst;
				rgrgiNeighbors[k][0 + rgcNeighbors[k]] = pShape->m_pReps[pShape->m_pFaces[i].m_pIndices[(pShape->m_pFaces[i].m_cIndices + j - 1) % pShape->m_pFaces[i].m_cIndices]].m_iFirst;
				rgrgiNeighbors[k][1 + rgcNeighbors[k]] = pShape->m_pReps[pShape->m_pFaces[i].m_pIndices[(pShape->m_pFaces[i].m_cIndices + j + 1) % pShape->m_pFaces[i].m_cIndices]].m_iFirst;
				rgcNeighbors[k] += 2;
			}
		}

		for (i = 0; i < pShape->m_cPoints; i++)
			rgiIndices[i] = i;
										// "good" vertices are those which have non-zero sum weights
		iBad = 0;						// "bad" vertices are those which have 0 sum weights but have a "good" neighbor
		iHorrible = pShape->m_cPoints;	// "horrible" vertices are those which have 0 sum weights and have no "good" neighbor

		// arrange vertices in the order of GOOD...BAD...HORRIBLE, and work from left to right
		while (iBad < pShape->m_cPoints)
		{
			if (iBad == iHorrible)
				iHorrible = pShape->m_cPoints;
			
			i = rgiIndices[iBad];

			if (0.0f != rgfVertWtSums[i])
			{
				iBad++;
				continue;
			}


			for (fDistSq = 0.0f, k = 0; k < 3; k++)
			{
				fDelta = pShape->m_pPoints[i][k] - pShape->m_pPoints[rgrgiNeighbors[i][0]][k]; 
				fDistSq += fDelta * fDelta;
			}
			iNearestGood = pShape->m_cPoints;
			fLeastDistSq = fDistSq + 1.0f;
			// find the nearest "good" neighbor that has weights
			for (j = 0; j < rgcNeighbors[i]; j++)
			{
				if (0.0f == rgfVertWtSums[rgrgiNeighbors[i][j]])
					continue;

				for (fDistSq = 0.0f, k = 0; k < 3; k++)
				{
					fDelta = pShape->m_pPoints[i][k] - pShape->m_pPoints[rgrgiNeighbors[i][j]][k];
					fDistSq += fDelta * fDelta;
				}

				if (fDistSq < fLeastDistSq)
				{
					fLeastDistSq = fDistSq;
					iNearestGood = rgrgiNeighbors[i][j];
				}
			}

			if (iNearestGood == pShape->m_cPoints)
			{
				iHorrible--;
				rgiIndices[iBad] = rgiIndices[iHorrible];
				rgiIndices[iHorrible] = i;
				continue;
			}

			for (j = 0; j < pShape->m_cBones; j++)
				rgfBoneVertWts[j * pShape->m_cPoints + i] = rgfBoneVertWts[j * pShape->m_cPoints + iNearestGood];
			rgfVertWtSums[i] = rgfVertWtSums[iNearestGood];

			iBad++;
		}

		for (i = 0; i < pShape->m_cBones; i++)
		{
			DWORD cWeights;

			for (cWeights = 0, j = 0; j < pShape->m_cPoints; j++)
			{
				if (0.0f != rgfBoneVertWts[i * pShape->m_cPoints + j])
				{
					rgfWeights[cWeights] = rgfBoneVertWts[i * pShape->m_cPoints + j];
					rgiIndices[cWeights] = j;
					cWeights++;
				}
			}

			delete[] pShape->m_pBones[i].m_pfWeights;
			delete[] pShape->m_pBones[i].m_piPoints;
			pShape->m_pBones[i].m_pfWeights = NULL;
			pShape->m_pBones[i].m_piPoints = NULL;

			if (NULL == (pShape->m_pBones[i].m_pfWeights = new FLOAT[cWeights]) ||
				NULL == (pShape->m_pBones[i].m_piPoints = new DWORD[cWeights]))
			{
				hr = E_OUTOFMEMORY;
				goto e_ExitKAYA;
			}

			memcpy(pShape->m_pBones[i].m_pfWeights, rgfWeights, cWeights * sizeof(FLOAT));
			memcpy(pShape->m_pBones[i].m_piPoints, rgiIndices, cWeights * sizeof(DWORD));
			pShape->m_pBones[i].m_cWeights = cWeights;
		}


e_ExitKAYA:
		delete[] rgiIndices;
		delete[] rgfWeights;
		delete[] rgfVertWtSums;
		delete[] rgfBoneVertWts;
		delete[] rgcNeighbors;
		if (NULL != rgrgiNeighbors)
			for (i = 0; i < pShape->m_cPoints; i++)
				delete[] rgrgiNeighbors[i];
		delete[] rgrgiNeighbors;

		if (FAILED(hr))
			goto e_Exit;
	}
#endif
#endif

	// ensure that weights sum to 1 by adding, if necessary, an EXTRA BONE (corresponding to this mesh's transform)
	if (pShape->m_cBones > 0)
	{
		DWORD nWeights = 0;
		DWORD nReps = 0;
		
		afWeights = NULL;
		if (NULL == (afWeights = new float[pShape->m_cPoints]))
		{
			hr = E_OUTOFMEMORY;
			cout << "LoadMesh(): Could not allocate extra bone's weight array." << endl;
			goto e_Exit;
		}
		memset(afWeights, 0, pShape->m_cPoints * sizeof(float));	// zero out array

		aiPoints = NULL;
		if (NULL == (aiPoints = new DWORD[pShape->m_cPoints]))
		{
			hr = E_OUTOFMEMORY;
			cout << "LoadMesh(): Could not allocate extra bone's point index array." << endl;
			goto e_Exit;
		}

		abBonePointArray = NULL;
		if (NULL == (abBonePointArray = new bool[pShape->m_cPoints]))
		{
			hr = E_OUTOFMEMORY;
			cout << "LoadMesh(): Could not allocate extra bone's point flag array." << endl;
			goto e_Exit;
		}
		memset(abBonePointArray, 0, pShape->m_cPoints * sizeof(bool));	// falsify array

		for (iBone = 0; iBone < pShape->m_cBones; iBone++)
		{
			for (iIndex = 0; iIndex < pShape->m_pBones[iBone].m_cWeights; iIndex++)
			{
				afWeights[pShape->m_pBones[iBone].m_piPoints[iIndex]] += pShape->m_pBones[iBone].m_pfWeights[iIndex];
			}
		}

		for (nWeights = 0, nReps = 0, iVert = 0; iVert < pShape->m_cPoints; iVert++)
		{
			if (afWeights[iVert] < 1.0f)
			{
				aiPoints[nWeights] = iVert;
				afWeights[nWeights] = 1.0f - afWeights[iVert];
				nReps += pShape->m_pReps[iVert].m_cReps;
				nWeights++;
				acBonesPerPoint[iVert]++;
				abBonePointArray[iVert] = true;
			}
		}

		if (nWeights > 0)
		{
			// get bone name
			delete[] szBone;
			szBone = NULL;
			szBone = new char[1 + mdpTransform.partialPathName().length()];
			if (NULL == szBone) 
			{
				hr = E_OUTOFMEMORY;
				cout << "LoadMesh(): Could not allocate bone name." << endl;
				goto e_Exit;
			}

			strcpy(szBone, mdpTransform.partialPathName().asChar());

			for (pc = szBone; *pc != '\0'; pc++)	
			{	// replace '|' and ' ' characters by '_' in the maya partial pathname
				if (*pc == ' ' || *pc == '|')
					*pc = '_';
			}

			iBoneIdx = pShape->m_cBones;

			// new bone
			if (iBoneIdx == cBonesMax)
			{	// double array size
				SBone* aBones;
				bool** aabBPT;

				cBonesMax *= 2;

				aBones = NULL;
				aBones = new SBone[cBonesMax];
				if (NULL == aBones)
				{
					cout << "LoadMesh(): Could not allocate bones." << endl;
					goto e_Exit;
				}

				memcpy(aBones, pShape->m_pBones, pShape->m_cBones * sizeof(SBone));

				aabBPT = NULL;
				aabBPT = new bool*[cBonesMax];
				if (NULL == aabBPT)
				{
					cout << "LoadMesh(): Could not allocate bones." << endl;
					goto e_Exit;
				}

				memset(aabBPT, 0, cBonesMax * sizeof(bool*));	// NULLify the table
				memcpy(aabBPT, aabBonePointTable, pShape->m_cBones * sizeof(bool*));

				delete[] pShape->m_pBones;
				delete[] aabBonePointTable;

				pShape->m_pBones = aBones;
				aabBonePointTable = aabBPT;
			}


			if (iBoneIdx == pShape->m_cBones)
			{	// initialize new bone
				// save bone name
				pShape->m_pBones[iBoneIdx].m_szName = szBone;
				g_Arrays.Add(pShape->m_pBones[iBoneIdx].m_szName);
				szBone = NULL;	// so that we won't later delete it

				pShape->m_cBones++;
			}

			pShape->m_pBones[iBoneIdx].m_cWeights = nWeights;
			pShape->m_pBones[iBoneIdx].m_cReps = nReps;

			pShape->m_pBones[iBoneIdx].m_pfWeights = afWeights;
			afWeights = NULL;

			pShape->m_pBones[iBoneIdx].m_piPoints = aiPoints;
			aiPoints = NULL;

			aabBonePointTable[iBoneIdx] = abBonePointArray;
			abBonePointArray = NULL;
		}
	}


	// NUM REPS PER BONE
	for (iBone = 0; iBone < pShape->m_cBones; iBone++)
	{
		pShape->m_pBones[iBone].m_cReps = 0;

		for (iVert = 0; iVert < pShape->m_pBones[iBone].m_cWeights; iVert++)
			pShape->m_pBones[iBone].m_cReps += pShape->m_pReps[pShape->m_pBones[iBone].m_piPoints[iVert]].m_cReps;
	}

	// MAX BONES PER POINT
	pShape->m_cMaxBonesPerPoint = 0;
	for (iVert = 0; iVert < pShape->m_cPoints; iVert++)
	{
		if (pShape->m_cMaxBonesPerPoint < acBonesPerPoint[iVert])
			pShape->m_cMaxBonesPerPoint = acBonesPerPoint[iVert];
	}

	// MAX BONES PER FACE
	pShape->m_cMaxBonesPerFace = 0;
	for (iFace = 0; iFace < pShape->m_cFaces; iFace++)
	{
		DWORD cBonesPerFace = 0;

		for (iBone = 0; iBone < pShape->m_cBones; iBone++)
		{
			for (iIndex = 0; iIndex < pShape->m_pFaces[iFace].m_cIndices; iIndex++)
			{
				if (aabBonePointTable[iBone][pShape->m_pReps[pShape->m_pFaces[iFace].m_pIndices[iIndex]].m_iFirst])
				{
					cBonesPerFace++;
					break;
				}
			}
		}

		if (pShape->m_cMaxBonesPerFace < cBonesPerFace)
			pShape->m_cMaxBonesPerFace = cBonesPerFace;
	}

   
    pShape->m_kType	= SShape::MESH;		// mesh type

	cout << "\t\t\t" 
		<< pShape->m_cReps << " rep(s), "
		<< pShape->m_cPoints << " point(s), "
		<< pShape->m_cNormals << " normal(s), "
		<< pShape->m_cUVs << " UV(s), "
		<< pShape->m_cGroups << " material(s), "
		<< pShape->m_cFaces << " face(s), "
		<< pShape->m_cBones << " bone(s), "
		<< pShape->m_cMaxBonesPerPoint << " max bones/point, "
		<< pShape->m_cMaxBonesPerFace << " max bones/face."
		<< endl;

	
e_Exit:

	if (NULL != aabBonePointTable)
	{
		for (iBone = 0; iBone < cBonesMax; iBone++)
			SAFE_DELETE_ARRAY(aabBonePointTable[iBone]);
	}

    SAFE_DELETE_ARRAY(aabBonePointTable);
	SAFE_DELETE_ARRAY(acBonesPerPoint);
	SAFE_DELETE_ARRAY(szBone);
	SAFE_DELETE_ARRAY(amtmBonePositions);
	SAFE_DELETE_ARRAY(afWeights);
	SAFE_DELETE_ARRAY(aiPoints);
	SAFE_DELETE_ARRAY(abBonePointArray);
    SAFE_DELETE_ARRAY(piNormalRemap);
    SAFE_DELETE_ARRAY(piUVRemap);

	return	hr;
}



//-----------------------------------------------------------------------------
// Name: IsPatchMesh()
// Desc: Checks if the NURBS surface is a bicubic Bezier quad-patch mesh
//-----------------------------------------------------------------------------

bool	IsPatchMesh
		(
			MDagPath& mdpNurbs
		) 
{
	MFnNurbsSurface	fnNurbs;

	if (!mdpNurbs.hasFn(MFn::kNurbsSurface))
		return false;		// not a nurbs surface

	if (!fnNurbs.setObject(mdpNurbs))
		return false;

	if (fnNurbs.degreeU() != 3 || fnNurbs.degreeV() != 3)	
		return false;		// not a bicubic surface

	int	kFormInU = fnNurbs.formInU();
	int	kFormInV = fnNurbs.formInV();

	if (kFormInU == MFnNurbsSurface::kInvalid || kFormInV == MFnNurbsSurface::kInvalid)
		return false;		// surface has invalid form

	if (kFormInU == MFnNurbsSurface::kPeriodic || kFormInV == MFnNurbsSurface::kPeriodic)
		return false;		// can't handle periodic surfaces
	
	int	cCVsInU	= fnNurbs.numCVsInU();
	int	cCVsInV	= fnNurbs.numCVsInV();

	if ((cCVsInU - 1) % 3 != 0 || (cCVsInV - 1) % 3 != 0)
		return false;		// invalid control point count (we only deal with quad patches)

	int	cSpansInU = (cCVsInU - 1) / 3;
	int	cSpansInV = (cCVsInV - 1) / 3;

	if (cSpansInU <= 0 || cSpansInV <= 0)
		return false;		// invalid span count

	return true;			// all tests passed
}



//-----------------------------------------------------------------------------
// Name: LoadSubd()
// Desc: Loads a Subdivision surface from a specified DAG path
//-----------------------------------------------------------------------------

HRESULT	LoadSubd
		(
			MDagPath&	mdpTransform, 
			MDagPath&	mdpSubd,
			SShape*		pShape
		) 
{
	HRESULT hr = S_OK;
	MStatus stat = MS::kSuccess;

	MFnTransform fnTransform;
	MFnSubd fnSubd;
	MFnMesh fnTess;

	MDagPath mdpMesh;
	MObject moMesh = MObject::kNullObj;

	if (!mdpTransform.hasFn(MFn::kTransform))
	{
		hr = E_FAIL;
		cout << "LoadSubd(): Input path is not a transform as expected.  Aborting..." << endl;
		goto e_Exit;
	}

	stat = fnTransform.setObject(mdpTransform);
	if (!stat)
	{
		hr = E_FAIL;
		cout << "LoadSubd(): Could not access transform object through function set.  Aborting..." << endl;
		goto e_Exit;
	}

	if (!mdpSubd.hasFn(MFn::kSubdiv))
	{
		hr = E_FAIL;
		cout << "LoadSubd(): Input path is not a SubD surface as expected.  Aborting..." << endl;
		goto e_Exit;
	}

	stat = fnSubd.setObject(mdpSubd);
	if (!stat)
	{
		hr = E_FAIL;
		cout << "LoadSubd(): Could not access SubD surface object through function set.  Aborting..." << endl;
		goto e_Exit;
	}

	if (!g_bExportSubd)
	{
		cout << "\t\tIgnoring SubD surface \"" << fnSubd.name().asChar() << "\"." << endl;
		goto e_Exit;
	}

	if (0 == g_dwSubdTess)
		goto e_SubdExport;

	// TESSELATE
	cout << "\t\tTesselating \"" << fnSubd.name().asChar() << "\"..." << endl;

	moMesh = fnSubd.tesselate((2 == g_dwSubdTess), 0, 0, mdpTransform.node(), &stat);
	if (!stat)
	{
		hr = E_FAIL;
		cout << "LoadSubd(): Could not tesselate SubD surface.  Aborting..." << endl;
		goto e_ExitTesselate;
	}

	if (!moMesh.hasFn(MFn::kMesh))
	{
		hr = E_FAIL;
		cout << "LoadSubd(): Tesselation did not produce a mesh.  Aborting..." << endl;
		goto e_ExitTesselate;
	}

	// rename tesselated mesh
	stat = fnTess.setObject(moMesh);
	if (!stat)
		cout << "WARNING: Could not access tesselated mesh to rename it." << endl;

	fnTess.setName(fnSubd.name() + "_TessX", &stat);
	if (!stat)
		cout << "WARNING: Failed to rename tesselated mesh." << endl;

	
	
	stat = MDagPath::getAPathTo(moMesh, mdpMesh);
	if (!stat)
	{
		hr = E_FAIL;
		cout << "LoadSubd(): Could not a DAG path to the tesselated mesh.  Aborting..." << endl;
		goto e_ExitTesselate;
	}

	hr = LoadMesh(mdpTransform, mdpMesh, pShape);
	if (FAILED(hr))
	{
		cout << "LoadSubd(): Could not load tesselated mesh.  Aborting..." << endl;
		goto e_ExitTesselate;
	}



e_ExitTesselate:

	if (!moMesh.isNull() && !fnTransform.object().isNull())
	{
		stat = fnTransform.removeChild(moMesh);
		if (!stat)
			cout << "WARNING: Could not remove tesselated mesh object." << endl;
	}

	goto e_Exit;


e_SubdExport:

	hr = E_NOTIMPL;
	cout << "LoadSubd(): Untesselated export not implemented.\n";
	goto e_Exit;

e_Exit:
	return hr;
}



//-----------------------------------------------------------------------------
// Name: LoadNURBS()
// Desc: Loads a NURBS surface from a specified DAG path
//-----------------------------------------------------------------------------

HRESULT	LoadNURBS
		(
			MDagPath&	mdpTransform, 
			MDagPath&	mdpNurbs,
			SShape*		pShape
		) 
{
	HRESULT hr = S_OK;
	MStatus stat = MS::kSuccess;



	MFnTransform fnTransform;
	MFnNurbsSurface fnNurbs;
	MFnMesh fnTess;

	MDagPath mdpMesh;
	MObject moMesh = MObject::kNullObj;


	if (!mdpTransform.hasFn(MFn::kTransform))
	{
		hr = E_FAIL;
		cout << "LoadNURBS(): Input path is not a transform as expected.  Aborting..." << endl;
		goto e_Exit;
	}

	stat = fnTransform.setObject(mdpTransform);
	if (!stat)
	{
		hr = E_FAIL;
		cout << "LoadNURBS(): Could not access transform object through function set.  Aborting..." << endl;
		goto e_Exit;
	}

	if (!mdpNurbs.hasFn(MFn::kNurbsSurface))
	{
		hr = E_FAIL;
		cout << "LoadNURBS(): Input path is not a NURBS surface as expected.  Aborting..." << endl;
		goto e_Exit;
	}

	stat = fnNurbs.setObject(mdpNurbs);
	if (!stat)
	{
		hr = E_FAIL;
		cout << "LoadNURBS(): Could not access NURBS surface object through function set.  Aborting..." << endl;
		goto e_Exit;
	}

	if (!g_bExportNURBS)
	{
		cout << "\t\tIgnoring NURBS surface \"" << fnNurbs.name().asChar() << "\"." << endl;
		goto e_Exit;
	}

	if (0 == g_dwNURBSTess)
		goto e_NURBSExport;
	else if (2 == g_dwNURBSTess)
		goto e_PatchExport;

	// TESSELATE
	cout << "\t\tTesselating \"" << fnNurbs.name().asChar() << "\"..." << endl;

	moMesh = fnNurbs.tesselate(MTesselationParams::fsDefaultTesselationParams, mdpTransform.node(), &stat);
	if (!stat)
	{
		hr = E_FAIL;
		cout << "LoadNURBS(): Could not tesselate NURBS surface.  Aborting..." << endl;
		goto e_ExitTesselate;
	}

	if (!moMesh.hasFn(MFn::kMesh))
	{
		hr = E_FAIL;
		cout << "LoadNURBS(): Tesselation did not produce a mesh.  Aborting..." << endl;
		goto e_ExitTesselate;
	}

	// rename tesselated mesh
	stat = fnTess.setObject(moMesh);
	if (!stat)
		cout << "WARNING: Could not access tesselated mesh to rename it." << endl;

	fnTess.setName(fnNurbs.name() + "_TessX", &stat);
	if (!stat)
		cout << "WARNING: Failed to rename tesselated mesh." << endl;

	stat = MDagPath::getAPathTo(moMesh, mdpMesh);
	if (!stat)
	{
		hr = E_FAIL;
		cout << "LoadNURBS(): Could not a DAG path to the tesselated mesh.  Aborting..." << endl;
		goto e_ExitTesselate;
	}

	hr = LoadMesh(mdpTransform, mdpMesh, pShape);
	if (FAILED(hr))
	{
		cout << "LoadNURBS(): Could not load tesselated mesh.  Aborting..." << endl;
		goto e_ExitTesselate;
	}


e_ExitTesselate:

	if (!moMesh.isNull() && !fnTransform.object().isNull())
	{
		stat = fnTransform.removeChild(moMesh);
		if (!stat)
			cout << "WARNING: Could not remove tesselated mesh object." << endl;
	}

	goto e_Exit;


e_NURBSExport:

	cout << "LoadNURBS(): Untesselated export not implemented.\n";
	hr = E_NOTIMPL;
	goto e_Exit;

e_PatchExport:

	cout << "LoadNURBS(): Patch export not implemented.\n";
	hr = E_NOTIMPL;
	goto e_Exit;

/*	
	// set up references




	// ensure that shape is a nurbs surface
	MObject	objNurb;

	DT_ATTEMPT(DtExt_ShapeGetShapeNode(iShape, objNurb));

	ASSERT(objNurb.hasFn(MFn::kNurbsSurface),	
				"Not a nurb surface");

	MFnNurbsSurface	fnNurb(objNurb);

	// ensure that surface is bicubic
	ASSERT(fnNurb.degreeU() == 3 && fnNurb.degreeV() == 3,	
				"Not a bicubic surface");

	// ensure correct form in U and V 
	int	kFormInU	= fnNurb.formInU();
	int	kFormInV	= fnNurb.formInV();

	ASSERT(kFormInU	== MFnNurbsSurface::kOpen || kFormInU == MFnNurbsSurface::kClosed,
				"Invalid form in U");

	ASSERT(kFormInV == MFnNurbsSurface::kOpen || kFormInV == MFnNurbsSurface::kClosed,
				"Invalid form in V");

	// ensure that surface is a quad mesh
	int	cCVsInU	= fnNurb.numCVsInU();
	int	cCVsInV	= fnNurb.numCVsInV();

	ASSERT((cCVsInU - 1) % 3 == 0 && (cCVsInV - 1) % 3 == 0,
				"Invalid CV count");

	// ensure that there is at least one patch
	int	cSpansInU	= (cCVsInU - 1) / 3;
	int	cSpansInV	= (cCVsInV - 1) / 3;

	ASSERT(cSpansInU > 0 && cSpansInV > 0, 
				"Invalid span count");


	// control vertices

	MPointArray rgCVs;

	fnNurb.getCVs(rgCVs);

	cVertices	= (int)rgCVs.length();
	rgVertices	= new DtVec3f[cVertices];

	ASSERT(rgVertices, 
				"Can't allocate memory for vertex array");

	for (DWORD iVertex = 0; iVertex < cVertices; iVertex++) 
	{
		rgVertices[iVertex].vec[0]	= (float)rgCVs[iVertex][0];
		rgVertices[iVertex].vec[1]	= (float)rgCVs[iVertex][1];
		rgVertices[iVertex].vec[2]	= (float)rgCVs[iVertex][2];
	}



	// texture coordinates
	cTexCoords	= (int)rgCVs.length();
	rgTexCoords	= new DtVec2f[cTexCoords];

	ASSERT(rgTexCoords, 
				"Can't allocate memory for texture coordinate array");

	for (int iCVInU = 0, iTexCoord = 0; iCVInU < cCVsInU; iCVInU++)
	{
		for (int iCVInV = 0; iCVInV < cCVsInV; iCVInV++, iTexCoord++)
		{
			rgTexCoords[iTexCoord].vec[0] =  ((float)iCVInU) / ((float)(cCVsInU - 1));
			rgTexCoords[iTexCoord].vec[1] =  ((float)iCVInV) / ((float)(cCVsInV - 1));
		}
	}



	// face info
	cFaces	= cSpansInU * cSpansInV;
	rgFaces	= new SFace[cFaces];

	ASSERT(rgFaces, 
				"Can't allocate memory for patch array");


	cFaceIndices	= 0;
			
	for (int iSpanInU = 0, iPatch = 0; iSpanInU < cSpansInU; iSpanInU++) 
	{
		int iCVIndexInU	= iSpanInU * 3;

		for (int iSpanInV = 0; iSpanInV < cSpansInV; iSpanInV++, iPatch++) 
		{
			int iCVIndexInV	= iSpanInV * 3;

			rgFaces[iPatch].m_cIndices		= 16;
            rgFaces[iPatch].m_pIndices		= new DWORD[rgFaces[iPatch].m_cIndices];

			ASSERT(rgFaces[iPatch].m_pIndices,
						"Could not allocate memory for patch indices");

            rgFaces[iPatch].m_pIndices[0]	= cCVsInV * (iCVIndexInU + 0) + (iCVIndexInV + 0);
            rgFaces[iPatch].m_pIndices[1]	= cCVsInV * (iCVIndexInU + 1) + (iCVIndexInV + 0);
			rgFaces[iPatch].m_pIndices[2]	= cCVsInV * (iCVIndexInU + 2) + (iCVIndexInV + 0);

            rgFaces[iPatch].m_pIndices[3]	= cCVsInV * (iCVIndexInU + 3) + (iCVIndexInV + 0);
			rgFaces[iPatch].m_pIndices[4]	= cCVsInV * (iCVIndexInU + 3) + (iCVIndexInV + 1);
			rgFaces[iPatch].m_pIndices[5]	= cCVsInV * (iCVIndexInU + 3) + (iCVIndexInV + 2);

			rgFaces[iPatch].m_pIndices[6]	= cCVsInV * (iCVIndexInU + 3) + (iCVIndexInV + 3);
			rgFaces[iPatch].m_pIndices[7]	= cCVsInV * (iCVIndexInU + 2) + (iCVIndexInV + 3);
			rgFaces[iPatch].m_pIndices[8]	= cCVsInV * (iCVIndexInU + 1) + (iCVIndexInV + 3);

			rgFaces[iPatch].m_pIndices[9]	= cCVsInV * (iCVIndexInU + 0) + (iCVIndexInV + 3);
			rgFaces[iPatch].m_pIndices[10]	= cCVsInV * (iCVIndexInU + 0) + (iCVIndexInV + 2);
			rgFaces[iPatch].m_pIndices[11]	= cCVsInV * (iCVIndexInU + 0) + (iCVIndexInV + 1);

			rgFaces[iPatch].m_pIndices[12]	= cCVsInV * (iCVIndexInU + 1) + (iCVIndexInV + 1);
			rgFaces[iPatch].m_pIndices[13]	= cCVsInV * (iCVIndexInU + 2) + (iCVIndexInV + 1);

			rgFaces[iPatch].m_pIndices[14]	= cCVsInV * (iCVIndexInU + 2) + (iCVIndexInV + 2);
			rgFaces[iPatch].m_pIndices[15]	= cCVsInV * (iCVIndexInU + 1) + (iCVIndexInV + 2);

			rgFaces[iPatch].m_iGroup		= 0;			//	WARNING: assumes only 1 material per surface


			cFaceIndices	+= rgFaces[iPatch].m_cIndices;
		}
	}



	// material info

	cGroups		= DtGroupGetCount(iShape);
	rgGroups	= new SGroup[cGroups];

	ASSERT(rgGroups, 
				"Can't allocate memory for material group array");


	ASSERT(cGroups == 1,
				"Assumption was made that NURBS surfaces have only 1 material");


	for (DWORD iGroup	= 0; iGroup < cGroups; iGroup++)
	{

		// material name
		DT_ATTEMPT(DtMtlGetName(iShape, iGroup, &rgGroups[iGroup].m_szMaterial));

		// texture file name
		DT_ATTEMPT(MyDtTextureGetFileName(rgGroups[iGroup].m_szMaterial, &rgGroups[iGroup].m_szTextureFile));

		// diffuse color 
		if (!rgGroups[iGroup].m_szTextureFile)
		{
			DT_ATTEMPT(DtMtlGetDiffuseClr(rgGroups[iGroup].m_szMaterial, 0, 
									   &rgGroups[iGroup].m_fDiffuseRed, 
									   &rgGroups[iGroup].m_fDiffuseGreen,
									   &rgGroups[iGroup].m_fDiffuseBlue));
		}
		else 	// material has a texture
		{
			//	load the diffuse factor into the diffuse components

			int		iMaterial;

			DT_ATTEMPT(DtMtlGetID(iShape, iGroup, &iMaterial));

			MObject	mShader;

			DT_ATTEMPT(DtExt_MtlGetShader(iMaterial, mShader));

			MFnLambertShader	fnShader;

			fnShader.setObject(mShader);

			float	fDiffuseFactor	= fnShader.diffuseCoeff();

			rgGroups[iGroup].m_fDiffuseRed	= fDiffuseFactor;
			rgGroups[iGroup].m_fDiffuseGreen	= fDiffuseFactor;
			rgGroups[iGroup].m_fDiffuseBlue	= fDiffuseFactor;
		}

		// specular color
		DT_ATTEMPT(DtMtlGetSpecularClr(rgGroups[iGroup].m_szMaterial, 0, 
									&rgGroups[iGroup].m_fSpecularRed, 
									&rgGroups[iGroup].m_fSpecularGreen, 
									&rgGroups[iGroup].m_fSpecularBlue));

		// emissive color
		DT_ATTEMPT(DtMtlGetEmissiveClr(rgGroups[iGroup].m_szMaterial, 0, 
									&rgGroups[iGroup].m_fEmissiveRed, 
									&rgGroups[iGroup].m_fEmissiveGreen, 
									&rgGroups[iGroup].m_fEmissiveBlue));

		// power / shininess
		DT_ATTEMPT(DtMtlGetShininess(rgGroups[iGroup].m_szMaterial, 0, &rgGroups[iGroup].m_fShininess));

		// transparency / alpha
		DT_ATTEMPT(DtMtlGetTransparency(rgGroups[iGroup].m_szMaterial, 0, &rgGroups[iGroup].m_fTransparency));
	}
	




	// vertex duplication info (very simple for patch meshes)

	cReps	= (int)rgCVs.length();
	rgReps	= new SRep[cReps];

	ASSERT(rgReps, 
				"Can't allocate memory for rep array");

	for (DWORD iRep = 0; iRep < cReps; iRep++) 
	{
		rgReps[iRep].m_iNorm	= -1;			// patches don't export normal info
		rgReps[iRep].m_iUV		= iRep;
		rgReps[iRep].m_iFirst	= iRep;
		rgReps[iRep].m_iNext	= iRep;

		rgReps[iRep].m_cReps	= 1;
	}



	// skinning info
			

	MObject	objShape;
	MObject objTransform;
	MObject	objInput;

	DT_ATTEMPT(DtExt_ShapeGetShapeNode(iShape, objShape));
	DT_ATTEMPT(DtExt_ShapeGetTransform(iShape, objTransform));




	// load the mesh's world transform (needed if skinning info is found)
	MDagPath	pathTransform;

	MFnDagNode(objTransform).getPath(pathTransform);

	MMatrix		matMeshWorldTransform	= pathTransform.inclusiveMatrix();





	cBones				= 0;
	rgBones				= NULL;

	cMaxBonesPerVertex	= 0;
	cMaxBonesPerFace	= 0;


	MObjectArray	rgobjBones;







	// smooth skinning

	bool*	rgbNonZeroFlagTable	= NULL;		// table of influences vs. vertices
	DWORD*	rgcNonZeros			= NULL;		// array of influence counts 

	bool bFoundSmoothSkin	= false;

	if (objShape.hasFn(MFn::kNurbsSurface)) 		// if this shape is a mesh
	{ 
		// loop through skin clusters
		for (MItDependencyNodes itSkin(MFn::kSkinClusterFilter); !itSkin.isDone(); itSkin.next()) 
		{
			MFnSkinCluster fnSkin(itSkin.item());

			// load input and output geometries
			MObjectArray	rgInputs;
			MObjectArray	rgOutputs;

			fnSkin.getInputGeometry(rgInputs);
			fnSkin.getOutputGeometry(rgOutputs);

			assert(rgInputs.length() == rgOutputs.length());		// ensure that input geometry count 
																	// equals output geometry count

			int	cInputs, cOutputs;

			cInputs	= cOutputs	= (int)rgOutputs.length();

			// loop through the output geometries
			for (int iOutput = 0, iInput = 0; iOutput < cOutputs; iOutput++, iInput++) 
			{
				assert(iOutput == iInput);		// sanity check

				
				if (rgOutputs[iOutput] == objShape) 		// if our shape is one of the output geometries
				{
					MDagPathArray	rgdagpathInfluences;
					
					cBones	= (int)fnSkin.influenceObjects(rgdagpathInfluences, &mStat);

					rgBones	= new SBone[cBones];

					ASSERT(rgBones,
								"Could not allocate memory for bone array");


					// initialize bones
					for (DWORD iBone = 0; iBone < cBones; iBone++) 
					{	// WARNING: not checking for new failure
						rgBones[iBone].m_szName			= new char[256];
						rgBones[iBone].m_cReps			= 0;
						rgBones[iBone].m_cWeights		= 0;
						rgBones[iBone].m_piPoints	= new int[cVertices];
						rgBones[iBone].m_pfWeights		= new float[cVertices];

						g_Strings.add(rgBones[iBone].m_szName);		// housekeeping

						// bone name
						strcpy(rgBones[iBone].m_szName, rgdagpathInfluences[iBone].partialPathName().asChar());

						// matrix offset
						MFnIkJoint fnBone(rgdagpathInfluences[iBone]);

						MObject objBindPose;

						fnBone.findPlug("bindPose").getValue(objBindPose);

						MFnMatrixData fnBindPose(objBindPose);

						(matMeshWorldTransform * fnBindPose.matrix().inverse()).get(rgBones[iBone].m_matOffset);



						rgobjBones.append(rgdagpathInfluences[iBone].node());
					}

					rgcNonZeros			= new DWORD[cVertices];

					ASSERT(rgcNonZeros,
								"Could not allocate memory for non zero count array");

					rgbNonZeroFlagTable	= new bool[cVertices * cBones];

					ASSERT(rgbNonZeroFlagTable,
								"Could not allocate memory for non zero table");

					// bone info; calculate max number of bones per vertex
					cMaxBonesPerVertex = 0;

					int iVertex = 0;

					MFnNurbsSurface fnOutput(rgOutputs[iOutput]);

					MDagPath dagpathOutputShape;
				
					fnOutput.getPath(dagpathOutputShape);

					// loop through the vertices
					for (MItGeometry itGeom(rgOutputs[iOutput]); !itGeom.isDone(); itGeom.next()) 
					{
	
						MFloatArray rgfWeights;

						unsigned cInfs;

						fnSkin.getWeights(dagpathOutputShape, itGeom.component(), rgfWeights, cInfs);

						int a = rgdagpathInfluences.length();
						int b = rgfWeights.length();
						int c = itGeom.count();

						assert(rgdagpathInfluences.length() == rgfWeights.length());
						assert(rgfWeights.length() == cInfs);

						rgcNonZeros[iVertex] = 0;


						float fWeightSum = 0.0f;

						for (DWORD iBone = 0; iBone < cBones; iBone++)
							fWeightSum += rgfWeights[iBone];

						assert(fWeightSum > 0.00001f);

						for (iBone = 0; iBone < cBones; iBone++) 
						{
							rgbNonZeroFlagTable[iBone * cVertices + iVertex]	= false;

							rgfWeights[iBone] = rgfWeights[iBone] / fWeightSum;		// normalize the weight

							if (rgfWeights[iBone] != 0.0f) 
							{
								rgcNonZeros[iVertex]++;

								rgBones[iBone].m_cReps += rgReps[iVertex].m_cReps;
		
								rgBones[iBone].m_piPoints[rgBones[iBone].m_cWeights]	= iVertex;							
								rgBones[iBone].m_pfWeights[rgBones[iBone].m_cWeights]	= rgfWeights[iBone];

								rgbNonZeroFlagTable[iBone * cVertices + iVertex]	= true;

								rgBones[iBone].m_cWeights++;
							}
						}


						if (rgcNonZeros[iVertex] > cMaxBonesPerVertex)
							cMaxBonesPerVertex = rgcNonZeros[iVertex];

						iVertex++;
					}



					// calculate max number of bones per vertex

					cMaxBonesPerFace	= 0;

					for (DWORD iFace = 0; iFace < cFaces; iFace++) 
					{
						DWORD	cBonesPerFace	= 0;

						for (DWORD iBone = 0; iBone < cBones; iBone++) 
						{
							for (DWORD iIndex = 0; iIndex < rgFaces[iFace].m_cIndices; iIndex++) 
							{
								if (rgbNonZeroFlagTable[iBone * cVertices + rgReps[rgFaces[iFace].m_pIndices[iIndex]].m_iFirst]) 
								{
									cBonesPerFace++;

									break;
								}
							}
						}

						if (cBonesPerFace > cMaxBonesPerFace)
							cMaxBonesPerFace = cBonesPerFace;
					}




					objInput = rgInputs[iInput];

					bFoundSmoothSkin = true;

					break;
				}
			}

			if (bFoundSmoothSkin)
				break;
		}
	}


	delete[] rgcNonZeros;
	delete[] rgbNonZeroFlagTable;



















	// rigid skinning

	rgbNonZeroFlagTable	= NULL;

	bool	bFoundRigidSkin	= false;

	if (!bFoundSmoothSkin) 	
	{
		cBones			= 1;						// zero'th bone is the extra "fake" bone
		DWORD cBonesMax	= 64;
		rgBones			= new SBone[cBonesMax];

		ASSERT(rgBones,
					"Could not allocate memory for bone array");

		rgbNonZeroFlagTable	= new bool[cBonesMax * cVertices];

		ASSERT(rgbNonZeroFlagTable,
					"Could not allocate memory for non-zero flag table");

		// fill non zero table with 0's
		memset(rgbNonZeroFlagTable, 0, cBonesMax * cVertices * sizeof(bool));

		// initialize "fake" iBone
		// WARNING: not checking for new failure
		rgBones[0].m_szName			= new char[256];
		rgBones[0].m_cReps			= 0;
		rgBones[0].m_cWeights		= 0;
		rgBones[0].m_pfWeights		= new float[cVertices];
		rgBones[0].m_piPoints	= new int[cVertices];

		g_Strings.add(rgBones[0].m_szName);							// housekeeping

		strcpy(rgBones[0].m_szName, SCENE_ROOT);					// bone name
	
		matMeshWorldTransform.get(rgBones[0].m_matOffset);			// "fake" bone has identity matrix


		// loop through joint clusters
		for (MItDependencyNodes itCluster(MFn::kJointCluster); !itCluster.isDone(); itCluster.next()) 
		{
			MFnWeightGeometryFilter fnCluster(itCluster.item());

			// load input and output geometries
			MObjectArray	rgInputs;
			MObjectArray	rgOutputs;

			fnCluster.getInputGeometry(rgInputs);
			fnCluster.getOutputGeometry(rgOutputs);

			assert(rgInputs.length() == rgOutputs.length());	// ensure input geometry count equals 
																// output geometry count

			int	cInputs, cOutputs;

			cInputs	= cOutputs	
				= (int)rgOutputs.length();

			// loop through the output geometries
			for (int iOutput = 0, iInput = 0; iOutput < cOutputs; iOutput++, iInput++) 
			{
				assert(iOutput == iInput);
				
				if (rgOutputs[iOutput] == objShape) 	// our shape is one of the output geometries
				{
					bFoundRigidSkin	= true;
		
					assert(rgInputs[iInput] == fnCluster.inputShapeAtIndex(iInput));	// sanity check

					objInput	= rgInputs[iInput];

					// get bone
					MPlug		plgMatrix	= fnCluster.findPlug("matrix", &mStat);

					MPlugArray	rgplgMatrixConnections;

					plgMatrix.connectedTo(rgplgMatrixConnections, true, false);			// get source plugs
					assert(rgplgMatrixConnections.length() == 1);

					MObject	objBone	= rgplgMatrixConnections[0].node();

					assert(objBone.hasFn(MFn::kJoint));

					MFnIkJoint fnBone(objBone);

					char	szBone[64];

					strcpy(szBone, fnBone.name().asChar());

					// find bone's index in current bone list
					for (DWORD iBone = 1; iBone < cBones;	iBone++) 
					{
						if (!strcmp(rgBones[iBone].m_szName, szBone))
							break;
					}
	
					if (iBone == cBones) 	// bone was not found in current bone list
					{
						// add bone
						if (cBones >= cBonesMax) 
						{
							// double array size
							cBonesMax  += cBonesMax;

							SBone*	rgNewBones	= new SBone[cBonesMax];

							ASSERT(rgNewBones, 
										"Could not allocate memory for new bone array");

							memcpy(rgNewBones, rgBones, cBones * sizeof(SBone));

							delete[] rgBones;

							rgBones	= rgNewBones;


							bool*	rgbNewNonZeroFlagTable	= new bool[cBonesMax * cVertices];

							ASSERT(rgbNewNonZeroFlagTable, 
										"Could not allocate memory for new non-zero flag table");

							memset(rgbNewNonZeroFlagTable, 0, cBonesMax * cVertices * sizeof(bool));
							memcpy(rgbNewNonZeroFlagTable, rgbNonZeroFlagTable, cBones * cVertices * sizeof(bool));

							delete[] rgbNonZeroFlagTable;

							rgbNonZeroFlagTable	= rgbNewNonZeroFlagTable;
						}
		
						// initialize iBone
						// WARNING: not checking for new failure
						rgBones[iBone].m_szName			= new char[256];
						rgBones[iBone].m_cReps			= 0;
						rgBones[iBone].m_cWeights		= 0;
						rgBones[iBone].m_piPoints	= new int[cVertices];
						rgBones[iBone].m_pfWeights		= new float[cVertices];

						g_Strings.add(rgBones[iBone].m_szName);							// housekeeping

						strcpy(rgBones[iBone].m_szName, szBone);						// bone name
	
						// matrix info
						MObject objBindPose;

						fnBone.findPlug("bindPose").getValue(objBindPose);

						MFnMatrixData fnBindPose(objBindPose);

						(matMeshWorldTransform * fnBindPose.matrix().inverse()).get(rgBones[iBone].m_matOffset);


						rgobjBones.append(objBone);

						cBones++;
					}



					char	szParent[64];

					bool	bFoundParent	= false;
					MObject	objParent; 

					for (DWORD iParent = 0; iParent < fnBone.parentCount(); iParent++) 
					{
						objParent	= fnBone.parent(iParent);

						MFnDagNode	fnParent(objParent);
						
						strcpy(szParent, fnParent.name().asChar());		// parent's name

						for (int iShape_ = 0; iShape_ < DtShapeGetCount(); iShape_++) 
						{
							char*	szShape;

							DT_ATTEMPT(DtShapeGetName(iShape_, &szShape));

							if (!strcmp(szParent, szShape)) 
							{
								bFoundParent	= true;

								break;
							}
						}

						if (bFoundParent)
							break;
					}
  
					iParent	= 0;

					if (bFoundParent) 	// parent shape found
					{
						// find parent bone's index in current bone list
						for (iParent = 1; iParent < cBones;	iParent++) 
						{
							if (!strcmp(rgBones[iParent].m_szName, szParent)) 
							{
								break;
							}
						}


						if (iParent == cBones) 		// parent bone was not found in current bone list
						{
							// add parent bone
							if (cBones >= cBonesMax) 
							{
								// double array size
								cBonesMax  += cBonesMax;

								SBone*	rgNewBones	= new SBone[cBonesMax];

								ASSERT(rgNewBones, 
											"Could not allocate memory for new bone array");
	
								memcpy(rgNewBones, rgBones, cBones * sizeof(SBone));
	
								delete[] rgBones;

								rgBones	= rgNewBones;

								
								bool*	rgbNewNonZeroFlagTable	= new bool[cBonesMax * cVertices];

								ASSERT(rgbNewNonZeroFlagTable, 
											"Could not allocate memory for new non-zero flag table");

								memset(rgbNewNonZeroFlagTable, 0, cBonesMax * cVertices * sizeof(bool));
								memcpy(rgbNewNonZeroFlagTable, rgbNonZeroFlagTable, cBones * cVertices * sizeof(bool));

								delete[] rgbNonZeroFlagTable;

								rgbNonZeroFlagTable	= rgbNewNonZeroFlagTable;
							}
			
							// initialize iBone
							// WARNING: not checking for new failure
							rgBones[iParent].m_szName		= new char[256];
							rgBones[iParent].m_cReps		= 0;
							rgBones[iParent].m_cWeights		= 0;
							rgBones[iParent].m_piPoints	= new int[cVertices];
							rgBones[iParent].m_pfWeights	= new float[cVertices];
	
							g_Strings.add(rgBones[iParent].m_szName);			// housekeeping
	
							strcpy(rgBones[iParent].m_szName, szParent);		// bone name

							// matrix info
							MObject	objBindPose;
	
							assert(objParent.hasFn(MFn::kJoint));

							MFnIkJoint(objParent).findPlug("bindPose").getValue(objBindPose);
	
							MFnMatrixData fnBindPose(objBindPose);
	
							(matMeshWorldTransform * fnBindPose.matrix().inverse()).get(rgBones[iParent].m_matOffset);


							rgobjBones.append(objParent);

							cBones++;
						}
					}


					// load weights
					MPlug		plgMessage	= fnCluster.findPlug("message");

					MPlugArray	rgplgMessageConnections;

					plgMessage.connectedTo(rgplgMessageConnections, false, true);	// get destination plugs

					assert(rgplgMessageConnections.length() == 1);
					assert(rgplgMessageConnections[0].node().hasFn(MFn::kSet));

					MFnSet fnSet(rgplgMessageConnections[0].node());
				
					MSelectionList list;

					fnSet.getMembers(list, false);

					assert(list.length() == 1);

					MDagPath	path;
					MObject		objComponents;

					list.getDagPath(0, path, objComponents);

					MFloatArray	rgWeights;

					fnCluster.getWeights(path, objComponents, rgWeights);

					assert(objComponents.hasFn(MFn::kDoubleIndexedComponent));


					MFnDoubleIndexedComponent fnComponent(objComponents);

					assert(fnComponent.elementCount() == (int)rgWeights.length());

					// loop through the weights
					for (int iWeight = 0; iWeight < (int)rgWeights.length(); iWeight++) 
					{
						assert(rgWeights[iWeight] <= 1.0f);

						int	iU, iV;

						fnComponent.getElement(iWeight, iU, iV);

						// WARNING: check calculation of iVertex
						int	iVertex	= iU * cCVsInV + iV;
						
						rgBones[iBone].m_pfWeights[rgBones[iBone].m_cWeights]	= rgWeights[iWeight];
						rgBones[iBone].m_piPoints[rgBones[iBone].m_cWeights]	= iVertex;

						rgBones[iBone].m_cReps	 += rgReps[iVertex].m_cReps;
						rgBones[iBone].m_cWeights++;

						rgbNonZeroFlagTable[iBone * cVertices + iVertex]	= true;

						if (rgWeights[iWeight] != 1.0f) 
						{
							rgBones[iParent].m_pfWeights[rgBones[iParent].m_cWeights]	= 1.0f - rgWeights[iWeight];
							rgBones[iParent].m_piPoints[rgBones[iParent].m_cWeights]	= iVertex;

							rgBones[iParent].m_cReps   += rgReps[iVertex].m_cReps;
							rgBones[iParent].m_cWeights++;				// IMPORTANT: Don't change line position

							rgbNonZeroFlagTable[iParent * cVertices + iVertex]	= true;
						}
					}

					break;
				}	// if found our mesh
			}	// loop thru geom's
		}	// loop thru joint clusters


		if (cBones == 1) 		// no rigid skinning found
		{
			delete[] rgBones[0].m_pfWeights;
			delete[] rgBones[0].m_piPoints;

			cBones	= 0;
		}
		else 
		{
			// at most 2 bones per vertex in rigid skinning (i.e. bone + parent)
			cMaxBonesPerVertex	= 2;

			// calculate max number of bones per vertex
			cMaxBonesPerFace	= 0;

			for (DWORD iFace = 0; iFace < cFaces; iFace++) 
			{
				DWORD	cBonesPerFace	= 0;

				for (DWORD iBone = 0; iBone < cBones; iBone++) 
				{
					for (DWORD iIndex = 0; iIndex < rgFaces[iFace].m_cIndices; iIndex++) 
					{
						if (rgbNonZeroFlagTable[iBone * cVertices + rgReps[rgFaces[iFace].m_pIndices[iIndex]].m_iFirst]) 
						{
							cBonesPerFace++;

							break;
						}
					}
				}

				if (cBonesPerFace > cMaxBonesPerFace)
					cMaxBonesPerFace = cBonesPerFace;
			}
			
		}
	}
				
				
	delete[] rgbNonZeroFlagTable;



















	// reload control vertices if skinning info was found

	if (bFoundSmoothSkin || bFoundRigidSkin)
	{
		delete[]	rgVertices;

		MyDtShapeGetControlPoints(objInput, objShape, &cVertices, &rgVertices);
	}





	// mesh type
	pShape->m_kType			= SShape::PATCH_MESH;

*/

e_Exit:
	return hr;	
}



//-----------------------------------------------------------------------------
// Name: AddSkin()
// Desc: Adds skinning info of a shape to an XFile data object
//-----------------------------------------------------------------------------

HRESULT	AddSkin
		(
			SShape*					pShape, 
			LPDIRECTXFILEDATA		pShapeDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		) 
{
	HRESULT	hr = S_OK;
	DWORD iBone;
	LPDIRECTXFILEDATA pSkinDataObject = NULL;   // skin data
	PBYTE pbSkinData = NULL;
	PBYTE pbSkinCurr;
	DWORD cbSkinSize;
	LPDIRECTXFILEDATA pBoneDataObject = NULL;   // bone data
	PBYTE pbBoneData = NULL;
	PBYTE pbBoneCurr;
	DWORD cbBoneSize;

    // allocate memory for skinmesh header
    cbSkinSize = sizeof(WORD)   // nMaxSkinWeightsPerVertex
               + sizeof(WORD)   // nMaxSkinWeightsPerFace
               + sizeof(WORD);  // nBones

	if (NULL == (pbSkinCurr = pbSkinData = new BYTE[cbSkinSize]))
	{
		hr = E_OUTOFMEMORY;
		goto e_Exit;
	}

	// nMaxSkinWeightsPerVertex
	WRITE(WORD, pbSkinCurr, ((WORD)pShape->m_cMaxBonesPerPoint));

	// nMaxSkinWeightsPerFace
	WRITE(WORD, pbSkinCurr, ((WORD)pShape->m_cMaxBonesPerFace));

	// nBones
	WRITE(WORD, pbSkinCurr, ((WORD)pShape->m_cBones));

    // create and add skin data object
	if (FAILED(hr = pxofSave->CreateDataObject(DXFILEOBJ_XSkinMeshHeader, NULL, NULL, cbSkinSize, pbSkinData, &pSkinDataObject)) ||
	    FAILED(hr = pShapeDataObject->AddDataObject(pSkinDataObject)))
		goto e_Exit;

	// SkinWeights
	for (iBone = 0; iBone < pShape->m_cBones; iBone++) 
	{
		DWORD iRow, iCol, iVertex, iRep;

        // allocate memory for bone
        cbBoneSize = sizeof(char*)                                      // transformNodeName
                   + sizeof(DWORD)                                      // nWeights
                   + sizeof(DWORD) * pShape->m_pBones[iBone].m_cReps    // vertexIndices[nWeights]
                   + sizeof(float) * pShape->m_pBones[iBone].m_cReps    // weights[nWeights]
                   + sizeof(float) * 16;                                // matrixOffset

        if (NULL == (pbBoneCurr = pbBoneData = new BYTE[cbBoneSize]))
		{
			hr = E_OUTOFMEMORY;
			goto e_Exit;
		}

		// transformNodeName
		WRITE(PCHAR, pbBoneCurr, ((char*)pShape->m_pBones[iBone].m_szName));

		// nWeights
		WRITE(DWORD, pbBoneCurr, ((DWORD)pShape->m_pBones[iBone].m_cReps));

		// vertexIndices[nWeights]
		for (iVertex = 0; iVertex < pShape->m_pBones[iBone].m_cWeights; iVertex++) 
		{
			iRep = pShape->m_pBones[iBone].m_piPoints[iVertex];

			do 
			{
				WRITE(DWORD, pbBoneCurr, ((DWORD)iRep));
				iRep = pShape->m_pReps[iRep].m_iNext;
			} while (iRep != pShape->m_pReps[iRep].m_iFirst);
		}

		// weights[nWeights]
		for (iVertex = 0; iVertex < pShape->m_pBones[iBone].m_cWeights; iVertex++)
			for (iRep = 0; iRep < pShape->m_pReps[pShape->m_pBones[iBone].m_piPoints[iVertex]].m_cReps; iRep++)
				WRITE(FLOAT, pbBoneCurr, pShape->m_pBones[iBone].m_pfWeights[iVertex]);

		// matrixOffset
		for (iRow = 0; iRow < 4; iRow++)
			for (iCol = 0; iCol < 4; iCol++)
				WRITE(FLOAT, pbBoneCurr, pShape->m_pBones[iBone].m_ppfOffset[iRow][iCol]);

        // create and add bone data object
		if (FAILED(hr = pxofSave->CreateDataObject(DXFILEOBJ_SkinWeights, NULL, NULL, cbBoneSize, pbBoneData, &pBoneDataObject)) ||
		    FAILED(hr = pShapeDataObject->AddDataObject(pBoneDataObject)))
			goto e_Exit;

        // release memory
		SAFE_DELETE_ARRAY(pbBoneData);
		SAFE_RELEASE(pBoneDataObject);
	}

e_Exit:
    // clean up
	SAFE_DELETE_ARRAY(pbSkinData);
	SAFE_DELETE_ARRAY(pbBoneData);
	SAFE_RELEASE(pSkinDataObject);
	SAFE_RELEASE(pBoneDataObject);

    if (FAILED(hr))
        cout << "AddSkin(): An error occured.\n";

	return hr;
}



//-----------------------------------------------------------------------------
// Name: AddNormals()
// Desc: Adds the normals of a shape to an XFile data object
//-----------------------------------------------------------------------------

HRESULT	AddNormals
		(
			SShape*					pShape, 
			LPDIRECTXFILEDATA		pShapeDataObject, 
			LPDIRECTXFILESAVEOBJECT pxofSave
		) 
{
	HRESULT	hr = S_OK;
	LPDIRECTXFILEDATA pNormalsDataObject = NULL;
	PBYTE pbNormalsData	= NULL;
	PBYTE pbNormalsCurr; 
	DWORD cbNormalsSize;
	DWORD iFace, iIndex, iNorm, iRep;

    if (0 == pShape->m_cNormals)
        goto EXIT;

    // allocate memory for normals
    cbNormalsSize = sizeof(DWORD)                                                 // nNormals
                  + pShape->m_cNormals * (3 * sizeof(float))                      // normals
                  + sizeof(DWORD)                                                 // nFaceNormals
                  + (pShape->m_cFaces + pShape->m_cFaceIndices) * sizeof(DWORD);  // faceNormals

    if (NULL == (pbNormalsCurr = pbNormalsData = new BYTE[cbNormalsSize]))
	{
		hr = E_OUTOFMEMORY;
		goto EXIT;
	}

	// nNormals
    WRITE(DWORD, pbNormalsCurr, ((DWORD)pShape->m_cNormals));

	// normals
	for (iNorm = 0; iNorm < pShape->m_cNormals; iNorm++) 
	{
		WRITE(FLOAT, pbNormalsCurr, pShape->m_pNormals[iNorm][0]);
		WRITE(FLOAT, pbNormalsCurr, pShape->m_pNormals[iNorm][1]);
		WRITE(FLOAT, pbNormalsCurr, pShape->m_pNormals[iNorm][2]);
	}



	// nFaceNormals
	WRITE(DWORD, pbNormalsCurr, ((DWORD)pShape->m_cFaces));

	// faceNormals
	for (iFace = 0; iFace < pShape->m_cFaces; iFace++) 
	{
		WRITE(DWORD, pbNormalsCurr, ((DWORD)pShape->m_pFaces[iFace].m_cIndices));

		for (iIndex = 0; iIndex < pShape->m_pFaces[iFace].m_cIndices; iIndex++)
        {
            iRep  = pShape->m_pFaces[iFace].m_pIndices[iIndex];
            iNorm = pShape->m_pReps[iRep].m_iNorm;

            WRITE(DWORD, pbNormalsCurr, iNorm);
        }
	}

    // create and add normals data object
	if (FAILED(hr = pxofSave->CreateDataObject(TID_D3DRMMeshNormals, NULL, NULL, cbNormalsSize, pbNormalsData, &pNormalsDataObject)) ||
	    FAILED(hr = pShapeDataObject->AddDataObject(pNormalsDataObject))) 
		goto EXIT;

EXIT:
	// clean up
	SAFE_DELETE_ARRAY(pbNormalsData);
	SAFE_RELEASE(pNormalsDataObject);

    if (FAILED(hr))
        cout << "AddNormals(): An error occured.\n";

	return hr;
}



//-----------------------------------------------------------------------------
// Name: AddUVs()
// Desc: Adds the UVs of a shape to an XFile data object
//-----------------------------------------------------------------------------

HRESULT	AddUVs
		(
			SShape*					pShape, 
			LPDIRECTXFILEDATA		pShapeDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		) 
{
	HRESULT	hr	= S_OK;
	DWORD iRep;
	LPDIRECTXFILEDATA pTexCoordsDataObject = NULL;
	PBYTE pbTexCoordsData = NULL;
	PBYTE pbTexCoordsCurr;
	DWORD cbTexCoordsSize;

    if (0 == pShape->m_cUVs)
        goto EXIT;

    // allocate memory for UVs
    cbTexCoordsSize = sizeof(DWORD)                           // nTextureCoords
                    + pShape->m_cReps * (2 * sizeof(float));  // textureCoords

	if (NULL == (pbTexCoordsCurr = pbTexCoordsData = new BYTE[cbTexCoordsSize]))
	{
		hr = E_OUTOFMEMORY;
		goto EXIT;
	}

	// nTextureCoords
	WRITE(DWORD, pbTexCoordsCurr, ((DWORD)pShape->m_cReps));

	// textureCoords
#ifdef KAYA
	if (!strcmp(g_szFace_c, pShape->m_szName))
	{
		const DOUBLE adCenter_c[3] = {0.0, 0.358, 0.386};	// center for spherical mapping
		const DOUBLE dUAngle_c = 170.377;
		const DOUBLE dUStart_c = 1.0 - dUAngle_c / 180.0;
		const DOUBLE dUCoeff_c = 90.0 / dUAngle_c;
		const DOUBLE dPi_c = 3.142592653589793238;
		DOUBLE x, y, z, w, u, v;

		for (iRep = 0; iRep < pShape->m_cReps; iRep++) 
		{
			x = (DOUBLE)pShape->m_pPoints[pShape->m_pReps[iRep].m_iFirst][0] - adCenter_c[0];
			y = (DOUBLE)pShape->m_pPoints[pShape->m_pReps[iRep].m_iFirst][1] - adCenter_c[1];
			z = (DOUBLE)pShape->m_pPoints[pShape->m_pReps[iRep].m_iFirst][2] - adCenter_c[2];

			w = x * x + z * z;
			z = z / sqrt(w);
			u = (asin(z) / dPi_c + 0.5) - dUStart_c;
			if (u >= 0.0)
				u *= dUCoeff_c;
			else 
				u = -u / dUStart_c;
			if (x > 0.0)
				u = 1.0 - u;

			y = y / sqrt(y * y + w);
			v = -(asin(y) / dPi_c + 0.5);

			WRITE(FLOAT, pbTexCoordsCurr, (FLOAT)u);
			WRITE(FLOAT, pbTexCoordsCurr, (FLOAT)v);
		}
	}
	else
#endif
	for (iRep = 0; iRep < pShape->m_cReps; iRep++) 
	{
		WRITE(FLOAT, pbTexCoordsCurr, g_fFlipU * pShape->m_pUVs[pShape->m_pReps[iRep].m_iUV][0]);
		WRITE(FLOAT, pbTexCoordsCurr, g_fFlipV * pShape->m_pUVs[pShape->m_pReps[iRep].m_iUV][1]);
	}

    // create and add texture coord data object
	if (FAILED(hr = pxofSave->CreateDataObject(TID_D3DRMMeshTextureCoords, NULL, NULL, cbTexCoordsSize, pbTexCoordsData, &pTexCoordsDataObject)) ||
	    FAILED(hr = pShapeDataObject->AddDataObject(pTexCoordsDataObject)))
		goto EXIT;


EXIT:
    // clean up
	SAFE_DELETE_ARRAY(pbTexCoordsData);
	SAFE_RELEASE(pTexCoordsDataObject);

    if (FAILED(hr))
        cout << "AddUVs(): An error occured.\n";

	return hr;
}



//-----------------------------------------------------------------------------
// Name: AddMaterialList()
// Desc: Adds the material list of a shape to an XFile data object
//-----------------------------------------------------------------------------

HRESULT	AddMaterialList
		(
			SShape*					pShape, 
			LPDIRECTXFILEDATA		pShapeDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		) 
{
	HRESULT	hr = S_OK;
	DWORD iGroup, iFace;
	LPDIRECTXFILEDATA pMaterialsDataObject = NULL;  // mesh material list
	PBYTE pbMaterialsData = NULL;
	PBYTE pbMaterialsCurr;
	DWORD cbMaterialsSize;
	LPDIRECTXFILEDATA pMaterialDataObject = NULL;   // material
	PBYTE pbMaterialData = NULL;
	PBYTE pbMaterialCurr;
	DWORD cbMaterialSize;
	LPDIRECTXFILEDATA pTextureDataObject = NULL;    // texture
    DWORD cbTextureSize;

	cbTextureSize = sizeof(char **);

    // allocate memory for material list
	cbMaterialsSize	= sizeof(DWORD)						// nMaterials
					+ sizeof(DWORD)						// nFaceIndexes
					+ pShape->m_cFaces * sizeof(DWORD);	// FaceIndexes

	if (NULL == (pbMaterialsCurr = pbMaterialsData = new BYTE[cbMaterialsSize]))
	{
		hr = E_OUTOFMEMORY;
		goto e_Exit;
	}

	// nMaterials
	WRITE(DWORD, pbMaterialsCurr, ((DWORD)pShape->m_cGroups));

	// nFaceIndexes
	WRITE(DWORD, pbMaterialsCurr, ((DWORD)pShape->m_cFaces));

	// FaceIndexes
	for (iFace = 0; iFace < pShape->m_cFaces; iFace++)
		WRITE(DWORD, pbMaterialsCurr, pShape->m_pFaces[iFace].m_iGroup);

	if (FAILED(hr = pxofSave->CreateDataObject(TID_D3DRMMeshMaterialList, NULL, NULL, cbMaterialsSize, pbMaterialsData, &pMaterialsDataObject)))
		goto e_Exit;

	// material data
	for (iGroup = 0; iGroup < pShape->m_cGroups; iGroup++) 
	{
		DWORD iRGB;

        // allocate memory for material
        cbMaterialSize = 4 * sizeof(float)    // faceColor
                       + sizeof(float)        // power
                       + 3 * sizeof(float)    // specularColor
                       + 3 * sizeof(float);   // emissiveColor

		if (NULL == (pbMaterialCurr = pbMaterialData = new BYTE[cbMaterialSize]))
		{
			hr = E_OUTOFMEMORY;
			goto e_Exit;
		}

		// faceColor
		for (iRGB = 0; iRGB < 3; iRGB++)
			WRITE(FLOAT, pbMaterialCurr, pShape->m_pGroups[iGroup].m_f3Diffuse[iRGB]);

		WRITE(FLOAT, pbMaterialCurr, (1.0f - pShape->m_pGroups[iGroup].m_fTransparency));

		// power
		WRITE(FLOAT, pbMaterialCurr, pShape->m_pGroups[iGroup].m_fShininess);
		
		// specularColor
		for (iRGB = 0; iRGB < 3; iRGB++)
			WRITE(FLOAT, pbMaterialCurr, pShape->m_pGroups[iGroup].m_f3Specular[iRGB]);

		// emissiveColor
		for (iRGB = 0; iRGB < 3; iRGB++)
			WRITE(FLOAT, pbMaterialCurr, pShape->m_pGroups[iGroup].m_f3Emissive[iRGB]);

		
		if (FAILED(hr = pxofSave->CreateDataObject(TID_D3DRMMaterial, pShape->m_pGroups[iGroup].m_szName, NULL, cbMaterialSize, pbMaterialData, &pMaterialDataObject)))
			goto e_Exit;


		// TextureFilename
#ifdef KAYA
        if (NULL != pShape->m_pGroups[iGroup].m_szName)
        {
            if (NULL == (pShape->m_pGroups[iGroup].m_szTextureFile = new char[1 + strlen(pShape->m_pGroups[iGroup].m_szName) + 3]))
            {
                hr = E_OUTOFMEMORY;
                goto e_Exit;
            }

            strcpy(pShape->m_pGroups[iGroup].m_szTextureFile, pShape->m_pGroups[iGroup].m_szName);
            strcat(pShape->m_pGroups[iGroup].m_szTextureFile, ".fx");
            g_Arrays.Add(pShape->m_pGroups[iGroup].m_szTextureFile);
        }
#endif
		if (NULL != pShape->m_pGroups[iGroup].m_szTextureFile) 
		{
			if (FAILED(hr = pxofSave->CreateDataObject(TID_D3DRMTextureFilename, NULL, NULL, cbTextureSize, &pShape->m_pGroups[iGroup].m_szTextureFile, &pTextureDataObject)) ||
			    FAILED(hr = pMaterialDataObject->AddDataObject(pTextureDataObject)))
				goto e_Exit;
		}
	    
		if (FAILED(hr = pMaterialsDataObject->AddDataObject(pMaterialDataObject)))
			goto e_Exit;

		SAFE_DELETE_ARRAY(pbMaterialData);
		SAFE_RELEASE(pMaterialDataObject);
		SAFE_RELEASE(pTextureDataObject);
	}
	
	if (FAILED(hr = pShapeDataObject->AddDataObject(pMaterialsDataObject)))
		goto e_Exit;

e_Exit:
    // clean up
	SAFE_DELETE_ARRAY(pbMaterialsData);
	SAFE_RELEASE(pMaterialsDataObject);
	SAFE_DELETE_ARRAY(pbMaterialData);
	SAFE_RELEASE(pMaterialDataObject);
	SAFE_RELEASE(pTextureDataObject);

    if (FAILED(hr))
        cout << "AddMaterialList(): An error occured.\n";
		
	return hr;
}



//-----------------------------------------------------------------------------
// Name: AddVertexColors()
// Desc: Adds the vertex colors of a shape to an XFile data object
//-----------------------------------------------------------------------------

HRESULT	AddVertexColors
		(
			SShape*					pShape, 
			LPDIRECTXFILEDATA		pShapeDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		) 
{
	HRESULT	hr = S_OK;
	LPDIRECTXFILEDATA pColorsDataObject = NULL;
	PBYTE pbColorsData = NULL;
	PBYTE pbColorsCurr;
	DWORD cbColorsSize;
	DWORD iRep;

    // allocate memory for vertex colors
    cbColorsSize = sizeof(DWORD)                                            // nVertexColors
                 + pShape->m_cReps * (sizeof(DWORD) + 4 * sizeof(float));   // vertexColors

	if (NULL == (pbColorsCurr = pbColorsData = new BYTE[cbColorsSize]))
	{
		hr = E_OUTOFMEMORY;
		goto e_Exit;
	}

	// nVertexColors
	WRITE(DWORD, pbColorsCurr, ((DWORD)pShape->m_cReps));

	// vertexColors
	for (iRep = 0; iRep < pShape->m_cReps; iRep++) 
	{
		// index
		WRITE(DWORD, pbColorsCurr, ((DWORD)iRep));

		// indexedColor
		WRITE(FLOAT, pbColorsCurr, pShape->m_pVertexColors[pShape->m_pReps[iRep].m_iFirst][0]);	// red
		WRITE(FLOAT, pbColorsCurr, pShape->m_pVertexColors[pShape->m_pReps[iRep].m_iFirst][1]);	// green
		WRITE(FLOAT, pbColorsCurr, pShape->m_pVertexColors[pShape->m_pReps[iRep].m_iFirst][2]);	// blue
		WRITE(FLOAT, pbColorsCurr, pShape->m_pVertexColors[pShape->m_pReps[iRep].m_iFirst][3]);	// alpha
	}

    // create and add vertex colors data object
	if (FAILED(hr = pxofSave->CreateDataObject(TID_D3DRMMeshVertexColors, NULL, NULL, cbColorsSize, pbColorsData, &pColorsDataObject)) ||
	    FAILED(hr = pShapeDataObject->AddDataObject(pColorsDataObject)))
		goto e_Exit;


e_Exit:
    // clean up
	SAFE_DELETE_ARRAY(pbColorsData);
	SAFE_RELEASE(pColorsDataObject);

    if (FAILED(hr))
        cout << "AddVertexColors(): An error occured.\n";

	return hr;
}



//-----------------------------------------------------------------------------
// Name: AddRepInfo()
// Desc: Adds the vertex duplication list of a shape to an XFile data object
//-----------------------------------------------------------------------------

HRESULT	AddRepInfo
		(
			SShape*					pShape, 
			LPDIRECTXFILEDATA		pShapeDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		) 
{
	HRESULT	hr = S_OK;
	LPDIRECTXFILEDATA pRepsDataObject = NULL;
	PBYTE pbRepsData = NULL;
	PBYTE pbRepsCurr;
	DWORD cbRepsSize;
	DWORD iRep;

    // allocate memory for vertex duplication indices
    cbRepsSize = sizeof(DWORD)                      // nIndices
               + sizeof(DWORD)                      // nOriginalVertices
               + pShape->m_cReps * sizeof(DWORD);   // indices

    if (NULL == (pbRepsCurr = pbRepsData = new BYTE[cbRepsSize]))
	{
		hr = E_OUTOFMEMORY;
		goto e_Exit;
	}

	// nIndices
	WRITE(DWORD, pbRepsCurr, ((DWORD)pShape->m_cReps));

	// nOriginalVertices
	WRITE(DWORD, pbRepsCurr, ((DWORD)pShape->m_cPoints));

	// indices
	for (iRep = 0; iRep < pShape->m_cReps; iRep++)
		WRITE(DWORD, pbRepsCurr, ((DWORD)pShape->m_pReps[iRep].m_iFirst));

    // create and add vertex duplication indices data object
	if (FAILED(hr = pxofSave->CreateDataObject(DXFILEOBJ_VertexDuplicationIndices, NULL, NULL, cbRepsSize, pbRepsData, &pRepsDataObject)) ||
	    FAILED(hr = pShapeDataObject->AddDataObject(pRepsDataObject)))
		goto e_Exit;

e_Exit:
    // clean up
	SAFE_DELETE_ARRAY(pbRepsData);
	SAFE_RELEASE(pRepsDataObject);

    if (FAILED(hr))
        cout << "AddReps(): An error occured.\n";

	return hr;
}



//-----------------------------------------------------------------------------
// Name: AddSubd()
// Desc: Adds a subdivision surface shape to XFile data object
//-----------------------------------------------------------------------------

HRESULT	AddSubD
		(
			SShape*					pShape, 
			LPDIRECTXFILEDATA		pFrameDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		) 
{
	HRESULT	hr = S_OK;

	hr = E_NOTIMPL;
	goto e_Exit;

e_Exit:
	return hr;
}



//-----------------------------------------------------------------------------
// Name: AddNURBS()
// Desc: Adds a NURBS surface shape to XFile data object
//-----------------------------------------------------------------------------

HRESULT	AddNURBS
		(
			SShape*					pShape, 
			LPDIRECTXFILEDATA		pFrameDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		) 
{
	HRESULT	hr = S_OK;

	hr = E_NOTIMPL;
	goto e_Exit;

e_Exit:
	return hr;
}



//-----------------------------------------------------------------------------
// Name: AddPatches()
// Desc: Adds a quad patch mesh shape to XFile data object
//-----------------------------------------------------------------------------

HRESULT	AddPatches
		(
			SShape*					pShape, 
			LPDIRECTXFILEDATA		pFrameDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		) 
{
	HRESULT	hr = S_OK;
	LPDIRECTXFILEDATA pShapeDataObject = NULL;
	PBYTE pbShapeData = NULL;
	DWORD iFace, iRep;	// counters
	DWORD cbShapeSize = sizeof(DWORD)													// nVertices
						+ pShape->m_cReps * (3 * sizeof(float))							// vertices
						+ sizeof(DWORD)													// nPatches
						+ (pShape->m_cFaces + pShape->m_cFaceIndices) * sizeof(DWORD);	// patches

	PBYTE pbShapeCurr = pbShapeData	= new BYTE[cbShapeSize];
	if (NULL == pbShapeData)
	{
		hr = E_OUTOFMEMORY;
		cout << "AddPatches(): Could not allocate memory for pbpShapeData." << endl;
		goto e_Exit;
	}
	

	// nVertices
	WRITE(DWORD, pbShapeCurr, ((DWORD)pShape->m_cPoints));

	// vertices
	for (iRep = 0; iRep < pShape->m_cReps; iRep++) 
	{
		WRITE(FLOAT, pbShapeCurr, pShape->m_pPoints[pShape->m_pReps[iRep].m_iFirst][0]);
		WRITE(FLOAT, pbShapeCurr, pShape->m_pPoints[pShape->m_pReps[iRep].m_iFirst][1]);
		WRITE(FLOAT, pbShapeCurr, pShape->m_pPoints[pShape->m_pReps[iRep].m_iFirst][2]);
	}

	// nPatches
	WRITE(DWORD, pbShapeCurr, ((DWORD)pShape->m_cFaces));

	// faces
	for (iFace = 0; iFace < pShape->m_cFaces; iFace++) 
	{
		WRITE(DWORD, pbShapeCurr, ((DWORD)16));

		for (DWORD iIndex = 0; iIndex < 16; iIndex++)
			WRITE(DWORD, pbShapeCurr, pShape->m_pFaces[iFace].m_pIndices[iIndex]);
	}


	hr = pxofSave->CreateDataObject(DXFILEOBJ_PatchMesh, NULL, NULL, cbShapeSize, pbShapeData, &pShapeDataObject);
	if (FAILED(hr))
	{
		cout << "AddPatches(): Could not create pShapeDataObject." << endl;
		goto e_Exit;
	}


	// MeshTextureCoords
	if (pShape->m_cUVs > 0) 
	{
		hr = AddUVs(pShape, pShapeDataObject, pxofSave);
		if (FAILED(hr))
			goto e_Exit;
	}

	// MeshVertexColors
//	hr = AddVertexColors(pShape, pShapeDataObject, pxofSave),
	if (FAILED(hr))
		goto e_Exit;
	
	// MeshMaterialList
	hr = AddMaterialList(pShape, pShapeDataObject, pxofSave);
	if (FAILED(hr))
		goto e_Exit;

	// VertexDuplicationIndices
	hr = AddRepInfo(pShape, pShapeDataObject, pxofSave);
	if (FAILED(hr))
		goto e_Exit;

	// XSkinMeshHeader
	if (pShape->m_cBones > 0) 
	{
		hr = AddSkin(pShape, pShapeDataObject, pxofSave);
		if (FAILED(hr))
			goto e_Exit;
	}

	hr = pFrameDataObject->AddDataObject(pShapeDataObject);
	if (FAILED(hr))
	{
		cout << "AddPatches(): Could not add pShapeDataObject to pFrameDataObject." << endl;
		goto e_Exit;
	}


e_Exit:
    // clean up
	SAFE_DELETE_ARRAY(pbShapeData);
	SAFE_RELEASE(pShapeDataObject);

	return hr;
}



//-----------------------------------------------------------------------------
// Name: AddMesh()
// Desc: Adds a poly mesh shape to XFile data object
//-----------------------------------------------------------------------------

HRESULT	AddMesh
		(
			SShape*					pShape, 
			LPDIRECTXFILEDATA		pFrameDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		) 
{

	HRESULT	hr = S_OK;
	LPDIRECTXFILEDATA pShapeDataObject = NULL;
	PBYTE pbMeshData = NULL;
	PBYTE pbMeshCurr;
	DWORD cbMeshSize;
	DWORD iFace, iRep;  // counters

    // allocate memory for mesh
    cbMeshSize = sizeof(DWORD)                                                  // nVertices
               + pShape->m_cReps * (3 * sizeof(float))                          // vertices
               + sizeof(DWORD)                                                  // nFaces
               + (pShape->m_cFaces + pShape->m_cFaceIndices) * sizeof(DWORD);   // faces

    if (NULL == (pbMeshCurr = pbMeshData = new BYTE[cbMeshSize]))
	{
		hr = E_OUTOFMEMORY;
		goto e_Exit;
	}

	// nVertices
	WRITE(DWORD, pbMeshCurr, ((DWORD)pShape->m_cReps));

	// vertices
	for (iRep = 0; iRep < pShape->m_cReps; iRep++) 
	{
		WRITE(FLOAT, pbMeshCurr, pShape->m_pPoints[pShape->m_pReps[iRep].m_iFirst][0]);
		WRITE(FLOAT, pbMeshCurr, pShape->m_pPoints[pShape->m_pReps[iRep].m_iFirst][1]);
		WRITE(FLOAT, pbMeshCurr, pShape->m_pPoints[pShape->m_pReps[iRep].m_iFirst][2]);
	}

	// nFaces
	WRITE(DWORD, pbMeshCurr, ((DWORD)pShape->m_cFaces));

	// faces
	for (iFace = 0; iFace < pShape->m_cFaces; iFace++) 
	{
		WRITE(DWORD, pbMeshCurr, ((DWORD)pShape->m_pFaces[iFace].m_cIndices));

		for (DWORD iIndex = 0; iIndex < pShape->m_pFaces[iFace].m_cIndices; iIndex++)
			WRITE(DWORD, pbMeshCurr, ((DWORD)pShape->m_pFaces[iFace].m_pIndices[iIndex]));
	}


    // create mesh data object
	if (FAILED(hr = pxofSave->CreateDataObject(TID_D3DRMMesh, pShape->m_szName, NULL, cbMeshSize, pbMeshData, &pShapeDataObject)))
		goto e_Exit;


	// MeshNormals
	if (pShape->m_cNormals > 0) 
	{
		if (FAILED(hr = AddNormals(pShape, pShapeDataObject, pxofSave)))
			goto e_Exit;
	}


	// MeshTextureCoords
	if (pShape->m_cUVs > 0) 
	{
		if (FAILED(hr = AddUVs(pShape, pShapeDataObject, pxofSave)))
			goto e_Exit;
	}

	//// MeshVertexColors (NOT YET TESTED... UNCOMMENT AT OWN RISK)
	//if (FAILED(hr = AddVertexColors(pShape, pShapeDataObject, pxofSave)))
	//	goto e_Exit;

	
	// MeshMaterialList
	if (pShape->m_cGroups > 0)
	{
		if (FAILED(hr = AddMaterialList(pShape, pShapeDataObject, pxofSave)))
			goto e_Exit;
	}


	// VertexDuplicationIndices
	if (FAILED(hr = AddRepInfo(pShape, pShapeDataObject, pxofSave)))
		goto e_Exit;


	// XSkinMeshHeader
	if (pShape->m_cBones > 0) 
	{
		if (FAILED(hr = AddSkin(pShape, pShapeDataObject, pxofSave)))
			goto e_Exit;
	}

	// add mesh data object
	if (FAILED(hr = pFrameDataObject->AddDataObject(pShapeDataObject)))
		goto e_Exit;

e_Exit:
    // clean up
	SAFE_DELETE_ARRAY(pbMeshData);
	SAFE_RELEASE(pShapeDataObject);

	if (FAILED(hr))
		cout << "AddMesh(): An error occured.\n";

	return hr;
}



//-----------------------------------------------------------------------------
// Name: AddShape()
// Desc: Adds a shape from the specified DAG path to XFile data object
//-----------------------------------------------------------------------------

HRESULT	AddShape
		(
			MDagPath&				mdpTransform,
			bool					bParentVisibility,
			LPDIRECTXFILEDATA		pParentFrameDataObject, 
			LPDIRECTXFILESAVEOBJECT	pxofSave
		) 
{
	MStatus	stat;
	SShape Shape;
	DWORD i;
	HRESULT	hr = S_OK;
	MItDag itDag;
	MFnTransform fnTransform;
	char *szName = NULL;
	bool bVisibility, bLodVisibility, bOverrideEnabled, bOverrideVisibility;
	bool bIsVisible;

    LPDIRECTXFILEDATA	pFrameDataObject	= NULL;

	if (!mdpTransform.hasFn(MFn::kTransform))
	{
		cout << "AddShape(): Object at DAG path was not a transform." << endl;
		goto e_Exit;
	}

	stat = fnTransform.setObject(mdpTransform.node());
	if (!stat)
	{
		cout << "AddShape(): Could not read transform object." << endl;
		goto e_Exit;
	}

	// check if mesh is visible
	bVisibility = true;
	bLodVisibility = true;
	bOverrideEnabled = false;
	bOverrideVisibility = true;

	do	// ONCE
	{
		MPlug mpVisibility, mpLodVisibility, mpOverrideEnabled, mpOverrideVisibility;

		mpVisibility = fnTransform.findPlug("visibility", &stat);
		if (!stat)
			break;

		stat = mpVisibility.getValue(bVisibility);
		if (!stat)
			break;

		mpLodVisibility = fnTransform.findPlug("lodVisibility", &stat);
		if (!stat)
			break;

		stat = mpLodVisibility.getValue(bLodVisibility);
		if (!stat)
			break;

		mpOverrideEnabled = fnTransform.findPlug("overrideEnabled", &stat);
		if (!stat)
			break;

		stat = mpOverrideEnabled.getValue(bOverrideEnabled);
		if (!stat)
			break;

		mpOverrideVisibility = fnTransform.findPlug("overrideVisibility", &stat);
		if (!stat)
			break;

		stat = mpOverrideVisibility.getValue(bOverrideVisibility);
		if (!stat)
			break;
	}
	while (false);

	bIsVisible = bParentVisibility && bLodVisibility && (!bOverrideEnabled || bOverrideVisibility);

	
	// shape name
	if (NULL == (szName = new char[1 + mdpTransform.partialPathName().length()])) 
	{
		hr = E_OUTOFMEMORY;
		goto e_Exit;
	}

	strcpy(szName, mdpTransform.partialPathName().asChar());
	for (i = 0; szName[i] != '\0'; i++)		// maya names may contain '|' or perhaps even ' '
	{
		if (szName[i] == ' ' || szName[i] == '|')
			szName[i] = '_';
	}
	
	if (FAILED(hr = g_Arrays.Add(szName)))
		goto e_Exit;

	cout << "\tReading \"" << mdpTransform.fullPathName().asChar() << "\"...";
	
	cout << "TRANSFORM";

	// Frame
	if (FAILED(hr = pxofSave->CreateDataObject(TID_D3DRMFrame, szName, NULL, 0, NULL, &pFrameDataObject)))
		goto e_Exit;
	
	if (FAILED(hr = AddTransform(mdpTransform, pFrameDataObject, pxofSave)))
		goto e_Exit;
	
	if (mdpTransform.hasFn(MFn::kJoint))
	{	// bone
		cout << "/BONE\n";
	}
	else if (bIsVisible && mdpTransform.hasFn(MFn::kMesh))
	{	// poly mesh
		MDagPath mdpMesh = mdpTransform;

		cout << "(MESH)" << endl;

		if (!mdpMesh.extendToShape())
		{
			hr = E_FAIL;
			goto e_Exit;
		}
		
		if (FAILED(hr = LoadMesh(mdpTransform, mdpMesh, &Shape)))
			goto e_Exit;
	}
	else if (bIsVisible && mdpTransform.hasFn(MFn::kNurbsSurface))
	{
		MDagPath mdpNurbs = mdpTransform;

		cout << "(NURBS)" << endl;
		
		if (!mdpNurbs.extendToShape())
		{
			hr = E_FAIL;
			goto e_Exit;
		}
		
		if (FAILED(hr = LoadNURBS(mdpTransform, mdpNurbs, &Shape)))
			goto e_Exit;
	}
	else if (bIsVisible && mdpTransform.hasFn(MFn::kSubdiv))
	{
		MDagPath mdpSubd = mdpTransform;

		cout << "(SUBD)" << endl;

		if (!mdpSubd.extendToShape())
		{
			hr = E_FAIL;
			goto e_Exit;
		}
		
		if (FAILED(hr = LoadSubd(mdpTransform, mdpSubd, &Shape)))
			goto e_Exit;
	}
	else
		cout << "\n";

	switch(Shape.m_kType)
	{
	case SShape::MESH:
        {
		    if (FAILED(hr = AddMesh(&Shape, pFrameDataObject, pxofSave)))
			    goto e_Exit;

		    break;
        }
	case SShape::SUBD:
	case SShape::NURBS:
	case SShape::PATCHES:
        {
		    hr = E_NOTIMPL;
		    goto e_Exit;

		    break;
        }
	default:
		break;
	};

	// add children
	itDag.reset(mdpTransform.node(), MItDag::kBreadthFirst, MFn::kTransform);

	itDag.next();

	for (; !itDag.isDone() && itDag.depth() == 1; itDag.next())
	{
		MDagPath mdpChild;

		itDag.getPath(mdpChild);

		if (!g_bIntermediateObjects_c && MFnDagNode(mdpChild).isIntermediateObject()) 
			continue;

		hr = AddShape(mdpChild, bIsVisible, pFrameDataObject, pxofSave);
		if (FAILED(hr))
			continue;
	}
		
	

	hr = pParentFrameDataObject->AddDataObject(pFrameDataObject);
	if (FAILED(hr))
		goto e_Exit;

	g_AddedPaths.append(mdpTransform);

e_Exit:
	if (pFrameDataObject)
		pFrameDataObject->Release();

	if (FAILED(hr))
		cout << "AddShape(): An error occured.\n";

	return hr;
}



//-----------------------------------------------------------------------------
// Name: LoadKeyframedAnims()
// Desc: Loads all keyframed animations
//-----------------------------------------------------------------------------

HRESULT LoadKeyframedAnims
		(
			DWORD*	pnAnims, 
			SAnim** paAnims
		)
{
	HRESULT	hr = S_OK;
	MStatus stat = MS::kSuccess;

	MAnimControl mAnimCtrl;
	MFnTransform fnTransform;
	SAnim* aAnims = NULL;
	SKey ParamKey;	// used to load the Euler rotation
	DWORD nFPS;	// num frames per second
	float fTimeFactor;
	DWORD nAnims = 0, nPending, nActive;
	DWORD iAnim, iPath, iPlug, iCurve, iKey, iPending;	// counters
	DWORD iRow, iCol;	// counters
	DWORD dwTemp;
	DWORD* aiCurKey = NULL;
	DWORD* aiIndices = NULL;
	MIntArray AnimPaths;
	float afMatrix[16];
	char* pcAnim;
	char* szAnim = NULL;
	MTime mtMinTime, mtOriginalTime;
	MTimeArray* amatTimes = NULL;

	mtOriginalTime = mAnimCtrl.currentTime();

	// calculate the frames per second
	switch(MTime::uiUnit()) 
	{
		case MTime::kSeconds:		// 1 fps
			nFPS = 1;
			break;
		case MTime::kMilliseconds:	// 1000 fps
			nFPS = 1000;
			break;
		case MTime::kGames:			// 15 fps
			nFPS = 15;
			break;
		case MTime::kFilm:			// 24 fps
			nFPS = 24;
			break;
		case MTime::kPALFrame:		// 25 fps
			nFPS = 25;
			break;
		case MTime::kNTSCFrame:		// 30 fps
			nFPS = 30;
			break;
		case MTime::kShowScan:		// 48 fps
			nFPS = 48;
			break;
		case MTime::kPALField:		// 50 fps
			nFPS = 50;
			break;
		case MTime::kNTSCField:		// 60 fps
			nFPS = 60;
			break;
		default:
			nFPS = 1;
			break;
	};

	fTimeFactor = 3600.0f / (float)nFPS;



	// find animated objects
	for (iPath = 0; iPath < g_AddedPaths.length(); iPath++)
	{
		MFnTransform fnTransform;

		stat = fnTransform.setObject(g_AddedPaths[iPath]);
		if (!stat)
			continue;

		if (!MAnimUtil::isAnimated(g_AddedPaths[iPath], false))
			continue;

		AnimPaths.append(iPath);
	}

	nAnims = AnimPaths.length();

	if (0 == nAnims)
		goto e_Exit;

	amatTimes = new MTimeArray[nAnims];
	if (NULL == amatTimes)
	{
		hr = E_OUTOFMEMORY;
		cout << "LoadKeyframedAnims(): Could not allocate times array." << endl;
		goto e_Exit;
	}

	aAnims = new SAnim[nAnims];
	if (NULL == aAnims)
	{
		hr = E_OUTOFMEMORY;
		cout << "LoadKeyframedAnims(): Could not allocate anim array." << endl;
		goto e_Exit;
	}

	aiCurKey = new DWORD[nAnims];
	if (NULL == aiCurKey)
	{
		hr = E_OUTOFMEMORY;
		cout << "LoadKeyframedAnims(): Could not allocate current-key array." << endl;
		goto e_Exit;
	}

	aiIndices = new DWORD[nAnims];
	if (NULL == aiIndices)
	{
		hr = E_OUTOFMEMORY;
		cout << "LoadKeyframedAnims(): Could not allocate indirection array." << endl;
		goto e_Exit;
	}

	// consolidate individual anim curves for each animated path
	for (nActive = 0, nPending = 0, iAnim = 0; iAnim < nAnims; iAnim++)
	{
		MPlugArray AnimPlugs;

		aiCurKey[iAnim] = 0;
		aiIndices[iAnim] = iAnim;

		MAnimUtil::findAnimatedPlugs(g_AddedPaths[AnimPaths[iAnim]], AnimPlugs, false, &stat);
		if (!stat)
		{
			cout << "LoadKeyframedAnims(): Ignoring animation because could not find animated plugs." << endl;
			continue;
		}

		for (iPlug = 0; iPlug < AnimPlugs.length(); iPlug++)
		{
			MObjectArray Curves;

			MAnimUtil::findAnimation(AnimPlugs[iPlug], Curves, &stat);
			if (!stat)
			{
				cout << "LoadKeyframedAnims(): Ignoring anim-plug because could not find animation." << endl;
				continue;
			}

			for (iCurve = 0; iCurve < Curves.length(); iCurve++)
			{
				DWORD iTime;
				MFnAnimCurve fnCurve(Curves[iCurve], &stat);
				if (!stat)
				{
					cout << "LoadKeyframedAnims(): Ignoring anim-curve because it could not be read with anim-curve function set." << endl;
					continue;
				}

				for (iTime = 0, iKey = 0; iKey < fnCurve.numKeys(); iKey++)
				{
					MTime mtTime = fnCurve.time(iKey, &stat);
					if (!stat)
					{
						cout << "LoadKeyframedAnims(): Ignoring anim-curve-key because it could not be read." << endl;
						continue;
					}

					for (; iTime < amatTimes[iAnim].length() && mtTime > amatTimes[iAnim][iTime]; iTime++);

					if (iTime < amatTimes[iAnim].length() && mtTime < amatTimes[iAnim][iTime])
					{
						amatTimes[iAnim].insert(mtTime, iTime);
					}
					else if (iTime == amatTimes[iAnim].length())
					{
						amatTimes[iAnim].append(mtTime);
					}
				}	// loop through keys
			}	// loop through curves
		}	// loop through plugs


		// get name
		delete[] szAnim;
		szAnim = NULL;
		szAnim = new char[g_AddedPaths[AnimPaths[iAnim]].partialPathName().length() + 1];
		if (NULL == szAnim)
		{
			hr = E_OUTOFMEMORY;
			cout << "AddAnimation(): Could not allocate string for anim name." << endl;
			goto e_Exit;
		}

		strcpy(szAnim, g_AddedPaths[AnimPaths[iAnim]].partialPathName().asChar());
		for (pcAnim = szAnim; *pcAnim != '\0'; pcAnim++)
		{
			if (*pcAnim == ' ' || *pcAnim == '|')
				*pcAnim = '_';
		}

		aAnims[iAnim].m_szName = szAnim;
		g_Arrays.Add(aAnims[iAnim].m_szName);
		szAnim = NULL;
		
		
		
		aAnims[iAnim].m_cKeys = amatTimes[iAnim].length();

		if (0 == aAnims[iAnim].m_cKeys)
			continue;

		aAnims[iAnim].m_pKeys = new SKey[aAnims[iAnim].m_cKeys];
		if (NULL == aAnims[iAnim].m_pKeys)
		{
			hr = E_OUTOFMEMORY;
			cout << "LoadKeyframedAnims(): Could not allocate anim-keys." << endl;
			goto e_Exit;
		}

		dwTemp = aiIndices[iAnim]; aiIndices[iAnim] = aiIndices[nPending]; aiIndices[nPending] = dwTemp;
		nPending++;
	}	// loop through animated paths



	while (nPending + nActive > 0)
	{
		iPending = 0;

		while (iPending < nPending)
		{
			iAnim = aiIndices[iPending];

			if (0 == nActive || amatTimes[iAnim][aiCurKey[iAnim]] <= mtMinTime)
			{
				mtMinTime = amatTimes[iAnim][aiCurKey[iAnim]];
				nPending--;
				nActive++;
				dwTemp = aiIndices[iPending]; aiIndices[iPending] = aiIndices[nPending]; aiIndices[nPending] = dwTemp;
				dwTemp = aiIndices[nAnims - nActive]; aiIndices[nAnims - nActive] = aiIndices[nPending]; aiIndices[nPending] = dwTemp;
				continue;
			}

			iPending++;
		}

		mAnimCtrl.setCurrentTime(mtMinTime);	// set time to mtMinTime

		while (nActive > 0)
		{
			iAnim = aiIndices[nAnims - nActive];
			
			if (mtMinTime < amatTimes[iAnim][aiCurKey[iAnim]])
			{
				mtMinTime = amatTimes[iAnim][aiCurKey[iAnim]];
				break;
			}

			// save current time
			aAnims[iAnim].m_pKeys[aiCurKey[iAnim]].m_iFrame = (DWORD)((float)mtMinTime.value() * fTimeFactor);

			// read matrix
			stat = fnTransform.setObject(g_AddedPaths[AnimPaths[iAnim]]);
			if (!stat)
			{
				hr = E_FAIL;
				cout << "LoadKeyframedAnims(): Could not read transform object using function set.  Aborting..." << endl;
				goto e_Exit;
			}

			for (iRow = 0; iRow < 4; iRow++)
				for (iCol = 0; iCol < 4; iCol++)
					afMatrix[iRow * 4 + iCol] = (float)fnTransform.transformation().asMatrix()[iRow][iCol];


			if (!MyMatrixGetTransforms(afMatrix, 
										aAnims[iAnim].m_pKeys[aiCurKey[iAnim]].m_f3Translation, 
										aAnims[iAnim].m_pKeys[aiCurKey[iAnim]].m_f3Scale,				
										aAnims[iAnim].m_pKeys[aiCurKey[iAnim]].m_f4Orientation,				
										aAnims[iAnim].m_pKeys[aiCurKey[iAnim]].m_f3Euler))
			{
				cout << "LoadKeyframedAnims(): Error in getting TRS components." << endl;
				goto e_Exit;
			}

			aAnims[iAnim].m_pKeys[aiCurKey[iAnim]].m_f4Orientation[0] = -aAnims[iAnim].m_pKeys[aiCurKey[iAnim]].m_f4Orientation[0];

			// get Euler Rotations & DOFs
			hr = LoadTransformKey(g_AddedPaths[AnimPaths[iAnim]], &ParamKey);
			if (FAILED(hr))
			{
				cout << "LoadKeyframedAnims(): Error in getting TRS components." << endl;
				goto e_Exit;
			}

			aAnims[iAnim].m_pKeys[aiCurKey[iAnim]].m_f3Euler[0] = ParamKey.m_f3Euler[0];
			aAnims[iAnim].m_pKeys[aiCurKey[iAnim]].m_f3Euler[1] = ParamKey.m_f3Euler[1];
			aAnims[iAnim].m_pKeys[aiCurKey[iAnim]].m_f3Euler[2] = ParamKey.m_f3Euler[2];

			aAnims[iAnim].m_pKeys[aiCurKey[iAnim]].m_pbDOFs[0] = ParamKey.m_pbDOFs[0];
			aAnims[iAnim].m_pKeys[aiCurKey[iAnim]].m_pbDOFs[1] = ParamKey.m_pbDOFs[1];
			aAnims[iAnim].m_pKeys[aiCurKey[iAnim]].m_pbDOFs[2] = ParamKey.m_pbDOFs[2];

#ifdef KAYA
			if (0 < ParamKey.m_cBlendWts)
			{
				if (NULL == (aAnims[iAnim].m_pKeys[aiCurKey[iAnim]].m_rgfBlendWts = new FLOAT[ParamKey.m_cBlendWts]))
				{
					hr = E_OUTOFMEMORY;
					goto e_Exit;
				}

				memcpy(aAnims[iAnim].m_pKeys[aiCurKey[iAnim]].m_rgfBlendWts, ParamKey.m_rgfBlendWts, ParamKey.m_cBlendWts * sizeof(FLOAT));
				aAnims[iAnim].m_pKeys[aiCurKey[iAnim]].m_cBlendWts = ParamKey.m_cBlendWts;
			}
#endif
			// update the current key
			aiCurKey[iAnim]++;

			if (aiCurKey[iAnim] < aAnims[iAnim].m_cKeys)
			{
				dwTemp = aiIndices[nAnims - nActive]; aiIndices[nAnims - nActive] = aiIndices[nPending]; aiIndices[nPending] = dwTemp;
				nPending++;
			}

			nActive--;
		} 
	}




e_Exit:
		

	mAnimCtrl.setCurrentTime(mtOriginalTime);

	if (FAILED(hr))
	{
		delete[] aAnims;
		nAnims = 0;
		aAnims = NULL;
	}

	delete[] amatTimes;
	delete[] aiCurKey;
	delete[] aiIndices;
	delete[] szAnim;

	*pnAnims = nAnims;
	*paAnims = aAnims;

	return hr;
}



//-----------------------------------------------------------------------------
// Name: LoadFixedStepAnims()
// Desc: Loads animations at a fixed step specified by g_iFrameStep
//-----------------------------------------------------------------------------

HRESULT LoadFixedStepAnims
		(
			DWORD*	pnAnims, 
			SAnim** paAnims
		)
{
	HRESULT	hr = S_OK;
	MStatus stat = MS::kSuccess;

	MAnimControl mAnimCtrl;
	MFnTransform fnTransform;
	SAnim* aAnims = NULL;
	SKey ParamKey;	// used to load the Euler rotation
	DWORD nFPS;	// num frames per second
	float fTimeFactor;
	DWORD nAnims = 0;
	DWORD iAnim, iPath, iKey;	// counters
	DWORD iRow, iCol;			// counters
	MIntArray AnimPaths;
	float afMatrix[16];
	char* pcAnim;
	char* szAnim = NULL;
	MTimeArray matKeys;
	MTime mtKey, mtMinTime, mtOriginalTime;

	mtOriginalTime = mAnimCtrl.currentTime();

	// calculate the frames per second
	switch(MTime::uiUnit()) 
	{
		case MTime::kSeconds:		// 1 fps
			nFPS = 1;
			break;
		case MTime::kMilliseconds:	// 1000 fps
			nFPS = 1000;
			break;
		case MTime::kGames:			// 15 fps
			nFPS = 15;
			break;
		case MTime::kFilm:			// 24 fps
			nFPS = 24;
			break;
		case MTime::kPALFrame:		// 25 fps
			nFPS = 25;
			break;
		case MTime::kNTSCFrame:		// 30 fps
			nFPS = 30;
			break;
		case MTime::kShowScan:		// 48 fps
			nFPS = 48;
			break;
		case MTime::kPALField:		// 50 fps
			nFPS = 50;
			break;
		case MTime::kNTSCField:		// 60 fps
			nFPS = 60;
			break;
		default:
			nFPS = 1;
			break;
	};

	fTimeFactor = 3600.0f / (float)nFPS;


	for (mtKey = MAnimControl::minTime(); mtKey <= MAnimControl::maxTime(); mtKey += g_iFrameStep)
		matKeys.append(mtKey);


	// find animated objects
	for (iPath = 0; iPath < g_AddedPaths.length(); iPath++)
	{
		MFnTransform fnTransform;

		stat = fnTransform.setObject(g_AddedPaths[iPath]);
		if (!stat)
			continue;

		if (!g_bAnimateEverything && !MAnimUtil::isAnimated(g_AddedPaths[iPath], false))
			continue;

		AnimPaths.append(iPath);
	}

	nAnims = AnimPaths.length();

	if (0 == nAnims)
		goto e_Exit;

	aAnims = new SAnim[nAnims];
	if (NULL == aAnims)
	{
		hr = E_OUTOFMEMORY;
		cout << "LoadFixedStepAnims(): Could not allocate anim array." << endl;
		goto e_Exit;
	}



	// consolidate individual anim curves for each animated path
	for (iAnim = 0; iAnim < nAnims; iAnim++)
	{
		// get name
		delete[] szAnim;
		szAnim = NULL;
		szAnim = new char[g_AddedPaths[AnimPaths[iAnim]].partialPathName().length() + 1];
		if (NULL == szAnim)
		{
			hr = E_OUTOFMEMORY;
			cout << "AddAnimation(): Could not allocate string for anim name." << endl;
			goto e_Exit;
		}

		strcpy(szAnim, g_AddedPaths[AnimPaths[iAnim]].partialPathName().asChar());
		for (pcAnim = szAnim; *pcAnim != '\0'; pcAnim++)
		{
			if (*pcAnim == ' ' || *pcAnim == '|')
				*pcAnim = '_';
		}

		aAnims[iAnim].m_szName = szAnim;
		g_Arrays.Add(aAnims[iAnim].m_szName);
		szAnim = NULL;
		
		
		
		aAnims[iAnim].m_cKeys = matKeys.length();
		if (0 == aAnims[iAnim].m_cKeys)
			continue;

		aAnims[iAnim].m_pKeys = new SKey[aAnims[iAnim].m_cKeys];
		if (NULL == aAnims[iAnim].m_pKeys)
		{
			hr = E_OUTOFMEMORY;
			cout << "LoadFixedStepAnims(): Could not allocate anim-keys." << endl;
			goto e_Exit;
		}
	}	// loop through animated paths


	for (iKey = 0; iKey < matKeys.length(); iKey++)
	{
		mAnimCtrl.setCurrentTime(matKeys[iKey]);	// update time

		for (iAnim = 0; iAnim < nAnims; iAnim++)
		{
			// set anim time
			aAnims[iAnim].m_pKeys[iKey].m_iFrame = (DWORD)((FLOAT)matKeys[iKey].value() * fTimeFactor);

			// read matrix
			stat = fnTransform.setObject(g_AddedPaths[AnimPaths[iAnim]]);
			if (!stat)
			{
				hr = E_FAIL;
				cout << "LoadFixedStepAnims(): Could not read transform object using function set.  Aborting..." << endl;
				goto e_Exit;
			}

			for (iRow = 0; iRow < 4; iRow++)
				for (iCol = 0; iCol < 4; iCol++)
					afMatrix[iRow * 4 + iCol] = (FLOAT)fnTransform.transformation().asMatrix()[iRow][iCol];


			if (!MyMatrixGetTransforms(afMatrix, 
										aAnims[iAnim].m_pKeys[iKey].m_f3Translation, 
										aAnims[iAnim].m_pKeys[iKey].m_f3Scale,				
										aAnims[iAnim].m_pKeys[iKey].m_f4Orientation,				
										aAnims[iAnim].m_pKeys[iKey].m_f3Euler))
			{
				cout << "LoadFixedStepAnims(): Error in getting TRS components." << endl;
				goto e_Exit;
			}

			aAnims[iAnim].m_pKeys[iKey].m_f4Orientation[0] = -aAnims[iAnim].m_pKeys[iKey].m_f4Orientation[0];

			// get Euler Rotations & DOFs
			hr = LoadTransformKey(g_AddedPaths[AnimPaths[iAnim]], &ParamKey);
			if (FAILED(hr))
			{
				cout << "LoadFixedStepAnims(): Error in getting TRS components." << endl;
				goto e_Exit;
			}

			aAnims[iAnim].m_pKeys[iKey].m_f3Euler[0] = ParamKey.m_f3Euler[0];
			aAnims[iAnim].m_pKeys[iKey].m_f3Euler[1] = ParamKey.m_f3Euler[1];
			aAnims[iAnim].m_pKeys[iKey].m_f3Euler[2] = ParamKey.m_f3Euler[2];

			aAnims[iAnim].m_pKeys[iKey].m_pbDOFs[0] = ParamKey.m_pbDOFs[0];
			aAnims[iAnim].m_pKeys[iKey].m_pbDOFs[1] = ParamKey.m_pbDOFs[1];
			aAnims[iAnim].m_pKeys[iKey].m_pbDOFs[2] = ParamKey.m_pbDOFs[2];
#ifdef KAYA
			if (0 < ParamKey.m_cBlendWts)
			{
				if (NULL == (aAnims[iAnim].m_pKeys[iKey].m_rgfBlendWts = new FLOAT[ParamKey.m_cBlendWts]))
				{
					hr = E_OUTOFMEMORY;
					goto e_Exit;
				}

				memcpy(aAnims[iAnim].m_pKeys[iKey].m_rgfBlendWts, ParamKey.m_rgfBlendWts, ParamKey.m_cBlendWts * sizeof(FLOAT));
				aAnims[iAnim].m_pKeys[iKey].m_cBlendWts = ParamKey.m_cBlendWts;
			}
#endif
		} // loop through animated objects

	} // loop through keys

e_Exit:
	// reset current time
	mAnimCtrl.setCurrentTime(mtOriginalTime);

	if (FAILED(hr))
	{	// cleanup in case of failure
		delete[] aAnims;
		nAnims = 0;
		aAnims = NULL;
	}

	delete[] szAnim;

	*pnAnims = nAnims;
	*paAnims = aAnims;

	return hr;
}



//-----------------------------------------------------------------------------
// CARRAYTABLE MEMBER FUNCTIONS
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Name: CArrayTable::CArrayTable()
// Desc: Contructor
//-----------------------------------------------------------------------------

CArrayTable::CArrayTable()
{
	m_cUsed = 0;
	m_ppbTable = NULL;
}



//-----------------------------------------------------------------------------
// Name: CArrayTable::~CArrayTable()
// Desc: Destructor
//-----------------------------------------------------------------------------

CArrayTable::~CArrayTable()
{
	Nuke();
}



//-----------------------------------------------------------------------------
// Name: CArrayTable::Init()
// Desc: Initializes the table of arrays
//-----------------------------------------------------------------------------

HRESULT CArrayTable::Init()
{
	if (NULL != m_ppbTable)
		return E_ACCESSDENIED;

	m_cUsed = 0;
	m_cSize = 1;

	if (NULL == (m_ppbTable = new PBYTE[m_cSize]))
		return E_OUTOFMEMORY;

	return S_OK;
}



//-----------------------------------------------------------------------------
// Name: CArrayTable::Nuke()
// Desc: Deletes all the arrays in the array table
//-----------------------------------------------------------------------------

HRESULT CArrayTable::Nuke()
{
	if (NULL != m_ppbTable)
	{
		for (DWORD i = 0; i < m_cUsed; i++)
		{
			delete[] m_ppbTable[i];		// delete all used entries in the table
		}
	}

	delete[] m_ppbTable;	// delete the entire table
	m_ppbTable = NULL;

	return S_OK;
}



//-----------------------------------------------------------------------------
// Name: CArrayTable::Add()
// Desc: Adds an array to the array table
//-----------------------------------------------------------------------------

HRESULT CArrayTable::Add(PVOID pb) 
{
	if (m_cUsed >= m_cSize) 
	{	// array is full, so grow it
		PBYTE* ppbTable = NULL;
		DWORD cNewSize = 1 + 2 * m_cSize;	// add 1 in case max size was zero

		if (NULL == (ppbTable = new PBYTE[cNewSize])) 
			return E_OUTOFMEMORY;	// not enough memory

		// copy old table into newly allocated table
		memcpy(ppbTable, m_ppbTable, m_cUsed * sizeof(PBYTE));

		delete[] m_ppbTable;	// delete the old table
		m_ppbTable = ppbTable;
		m_cSize = cNewSize;
	}

	m_ppbTable[m_cUsed] = (PBYTE)pb;
	m_cUsed++;

	return S_OK;
}



//-----------------------------------------------------------------------------
// SSHAPE MEMBER FUNCTIONS
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Name: SShape::SShape()
// Desc: Constructor
//-----------------------------------------------------------------------------

SShape::SShape() 
{
	m_kType = SShape::UNKNOWN;
	m_szName = NULL;

	m_cGroups = 0;
	m_pGroups = NULL;

	m_cPoints = 0;
	m_pPoints = NULL;

	m_cVertexColors = 0;
	m_pVertexColors = NULL;

	m_cNormals = 0;
	m_pNormals = NULL;

	m_cUVs = 0;
	m_pUVs = NULL;

	m_cReps = 0;
	m_pReps = NULL;

	m_cFaces = 0;
	m_pFaces = NULL;

    m_cFaceIndices = 0;

	m_cBones = 0;
	m_pBones = NULL;

	m_cMaxBonesPerFace = 0;
	m_cMaxBonesPerPoint = 0;
}



//-----------------------------------------------------------------------------
// Name: SShape::~SShape()
// Desc: Destructor
//-----------------------------------------------------------------------------

SShape::~SShape() 
{
	// WARNING: m_szName is NOT deleted here!  It should be deleted elsewhere.
	delete[] m_pReps;
	delete[] m_pPoints;
	delete[] m_pNormals;
	delete[] m_pUVs;
	delete[] m_pVertexColors;
	delete[] m_pFaces;
	delete[] m_pGroups;
	if (NULL != m_pBones)
	{
		for (DWORD i = 0; i < m_cBones; i++)
		{
			delete[] m_pBones[i].m_pfWeights;
			delete[] m_pBones[i].m_piPoints;
		}
	}
	delete[] m_pBones;
}



//-----------------------------------------------------------------------------
// SFACE MEMBER FUNCTIONS
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Name: SFace::SFace()
// Desc: Constructor
//-----------------------------------------------------------------------------

SFace::SFace()
{
	m_cIndices = 0;
	m_pIndices = NULL;
}



//-----------------------------------------------------------------------------
// Name: SFace::~SFace()
// Desc: Destructor
//-----------------------------------------------------------------------------

SFace::~SFace() 
{
	delete[] m_pIndices;
}



//-----------------------------------------------------------------------------
// SGROUP MEMBER FUNCTIONS
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Name: SGroup::SGroup()
// Desc: Constructor
//-----------------------------------------------------------------------------

SGroup::SGroup()
{
	m_f3Diffuse[0] = 0.5f;	// gray
	m_f3Diffuse[1] = 0.5f;
	m_f3Diffuse[2] = 0.5f;

	m_f3Emissive[0] = 0.0f;	// no emissive
	m_f3Emissive[1] = 0.0f;
	m_f3Emissive[2] = 0.0f;

	m_f3Specular[0] = 0.5f;	// gray
	m_f3Specular[1] = 0.5f;
	m_f3Specular[2] = 0.5f;

	m_fTransparency = 0.0f;	// opaque
	m_fShininess    = 1.0f;

	m_szName = NULL;
	m_szTextureFile = NULL;
}



//-----------------------------------------------------------------------------
// Name: SGroup::~SGroup()
// Desc: Destructor
//-----------------------------------------------------------------------------

SGroup::~SGroup() 
{
	// NOTE: m_szName and m_szTextureFile should be deleted elsewhere
}



//-----------------------------------------------------------------------------
// SBONE MEMBER FUNCTIONS
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Name: SBone::SBone()
// Desc: Constructor
//-----------------------------------------------------------------------------

SBone::SBone()
{
	DWORD iRow, iCol;

	m_szName = NULL;

	m_cReps = 0;

	m_cWeights = 0;
	m_pfWeights	= NULL;
	m_piPoints = NULL;

	// initialize bone offset matrix to identity
	for (iRow = 0; iRow < 4; iRow++)
	{
		for (iCol = 0; iCol < 4; iCol++)
			m_ppfOffset[iRow][iCol] = 0.0f;

		m_ppfOffset[iRow][iRow] = 1.0f;
	}
}



//-----------------------------------------------------------------------------
// Name: SBone::~SBone()
// Desc: Destructor
//-----------------------------------------------------------------------------

SBone::~SBone()
{
	// NOTE: m_pfWeights and m_piPoints are NOT deleted here.
	//       They should be deleted elsewhere.
}



//-----------------------------------------------------------------------------
// SANIM MEMBER FUNCTIONS
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Name: SAnim::SAnim()
// Desc: Constructor
//-----------------------------------------------------------------------------

SAnim::SAnim()
{
	m_szName = NULL;

	m_cKeys = 0;
	m_pKeys = NULL;
}



//-----------------------------------------------------------------------------
// Name: SAnim::~SAnim()
// Desc: Destructor
//-----------------------------------------------------------------------------

SAnim::~SAnim()
{
	delete[] m_pKeys;
}



#ifdef KAYA
//-----------------------------------------------------------------------------
// SKEY MEMBER FUNCTIONS
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Name: SKey::SKey()
// Desc: Constructor
//-----------------------------------------------------------------------------

SKey::SKey()
{
	m_cBlendWts = 0;
	m_rgfBlendWts = NULL;
}



//-----------------------------------------------------------------------------
// Name: SKey::~SKey()
// Desc: Destructor
//-----------------------------------------------------------------------------

SKey::~SKey()
{
	delete[] m_rgfBlendWts;
}
#endif






//-----------------------------------------------------------------------------
// USEFUL CODE
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// The following is old code salvaged from the Maya 3.0 exporter.
// It may be useful later.
//-----------------------------------------------------------------------------


/*




int	MyDtShapeGetControlPoints
	(
		MObject&	objInput, 
		MObject&	objOutput, 
		UINT*		pcVertices, 
		DtVec3f**	prgVertices
	) 
{
	*pcVertices		= 0;
	*prgVertices	= NULL;

	
	assert(objInput.hasFn(MFn::kNurbsSurface) && objOutput.hasFn(MFn::kNurbsSurface));


	MFnNurbsSurface	fnOutput(objOutput);
	MFnNurbsSurface	fnInput(objInput);



	MPointArray rgCVs;

	fnInput.getCVs(rgCVs);

	*pcVertices		= rgCVs.length();
	*prgVertices	= new DtVec3f[*pcVertices];

	if (!*prgVertices) 
	{
		*pcVertices	= 0;

		return 0;
	}

	// WARNING:  Is this homogeneous coordinates? Should I divide w?
	for (UINT iVertex = 0; iVertex < *pcVertices; iVertex++) 
	{
		(*prgVertices)[iVertex].vec[0]	= (float)rgCVs[iVertex][0];
		(*prgVertices)[iVertex].vec[1]	= (float)rgCVs[iVertex][1];
		(*prgVertices)[iVertex].vec[2]	= (float)rgCVs[iVertex][2];
	}


	// check for tweaks
	MPlug	plgTweakLoc	= fnOutput.findPlug("tweakLocation");

	MObject	objTweakLocVal;

	plgTweakLoc.getValue(objTweakLocVal);

	if (!objTweakLocVal.isNull())	// tweak found
	{	
		MPlugArray	rgplgTweakLocConnections;

		plgTweakLoc.connectedTo(rgplgTweakLocConnections, true, false);		// get source plugs

		assert(rgplgTweakLocConnections.length() == 1);

		MObject	objTweak = rgplgTweakLocConnections[0].node();

		assert(objTweak.hasFn(MFn::kTweak));

		MFnGeometryFilter	fnTweak(objTweak);

		bool	bRelativeTweak;

		fnTweak.findPlug("relativeTweak").getValue(bRelativeTweak);

		if (!bRelativeTweak) 
			cout << "\t\tWARNING: Encountered an absolute tweak; treating as relative!" << endl;

		MPlug plgOffsets = fnTweak.findPlug("plist")[0].child(0);


		//	WARNING: Seems like Maya doesn't initialize it's numElements properly!!
//		assert((int)plgOffsets.numElements() == cVertices);
		if ((int)plgOffsets.numElements() != *pcVertices)
			cout << "\t\tWARNING: tweak count doesn't match vertex count!" << endl;

		float	fEnvelope	= fnTweak.envelope();

		for (UINT iVertex = 0; iVertex < *pcVertices; iVertex++) 
		{
			DtVec3f	vecOffset;

			plgOffsets.elementByLogicalIndex(iVertex).child(0).getValue(vecOffset.vec[0]);
			plgOffsets.elementByLogicalIndex(iVertex).child(1).getValue(vecOffset.vec[1]);
			plgOffsets.elementByLogicalIndex(iVertex).child(2).getValue(vecOffset.vec[2]);

			(*prgVertices)[iVertex].vec[0]	+= fEnvelope * vecOffset.vec[0];
			(*prgVertices)[iVertex].vec[1]	+= fEnvelope * vecOffset.vec[1];
			(*prgVertices)[iVertex].vec[2]	+= fEnvelope * vecOffset.vec[2];
		}
	}
			
			

	return 1;
}


*/


/*

HRESULT	LoadFixedStepAnims
		(
			SAnim*	rgAnims
		)
{
	HRESULT	hr	= S_OK;

	cout << "\treading at intervals of " << g_iFrameStep << " frame(s)" << endl;

	// calculate the frames per second
	int	iFPS	= 1;

	switch(MTime::uiUnit()) 
	{
		case MTime::kSeconds:		// 1 fps
			iFPS	= 1;
			break;
		case MTime::kMilliseconds:	// 1000 fps
			iFPS	= 1000;
			break;
		case MTime::kGames:			// 15 fps
			iFPS	= 15;
			break;
		case MTime::kFilm:			// 24 fps
			iFPS	= 24;
			break;
		case MTime::kPALFrame:		// 25 fps
			iFPS	= 25;
			break;
		case MTime::kNTSCFrame:		// 30 fps
			iFPS	= 30;
			break;
		case MTime::kShowScan:		// 48 fps
			iFPS	= 48;
			break;
		case MTime::kPALField:		// 50 fps
			iFPS	= 50;
			break;
		case MTime::kNTSCField:		// 60 fps
			iFPS	= 60;
			break;
		default:
			iFPS	= 1;
			break;
	};

	float fTimeFactor	= 3600.0f / (float)iFPS;


	MTime	timeStart(MAnimControl::minTime().value(), MTime::uiUnit());
	MTime	timeEnd(MAnimControl::maxTime().value(), MTime::uiUnit());
	MTime	timeCurrent(MAnimControl::currentTime().value(), MTime::uiUnit());


	DtFrameSetStart((int)timeStart.value());
	DtFrameSetEnd((int)timeEnd.value());


	int cShapes	= DtShapeGetCount();


	MIntArray*	rgrgiKeys	= new MIntArray[cShapes];


	for (int iShape = 0; iShape < cShapes; iShape++) 
	{
		rgAnims[iShape].m_szName	= new char[256];
		rgAnims[iShape].m_cKeys		= 0;
		rgAnims[iShape].m_pKeys	= new SKey[1 + (DtFrameGetEnd() - DtFrameGetStart() + 1) / g_iFrameStep];

		g_Arrays.Add(STRING, rgAnims[iShape].m_szName);

		char* szName;

		DtShapeGetName(iShape, &szName);

		strcpy(rgAnims[iShape].m_szName, szName);

		DtShapeGetTRSAnimKeys(iShape, &rgrgiKeys[iShape]);
	}

	for (int iFrame = DtFrameGetStart(); iFrame <= DtFrameGetEnd(); iFrame += g_iFrameStep) 
	{
		DtFrameSet(iFrame);

		for (int iShape = 0; iShape < cShapes; iShape++) 
		{
			if (rgrgiKeys[iShape].length() > 0 || g_bAnimateEverything) 
			{
				rgAnims[iShape].m_pKeys[rgAnims[iShape].m_cKeys].m_iFrame	= (int)((float)iFrame * fTimeFactor);

				float*	rgfTRS;

				DtShapeGetMatrix(iShape, &rgfTRS);

				DtMatrixGetTransforms(rgfTRS, 
									  rgAnims[iShape].m_pKeys[rgAnims[iShape].m_cKeys].m_pfPosition, 
									  rgAnims[iShape].m_pKeys[rgAnims[iShape].m_cKeys].m_pfScale, 
									  rgAnims[iShape].m_pKeys[rgAnims[iShape].m_cKeys].m_pfQuaternion, 
									  rgAnims[iShape].m_pKeys[rgAnims[iShape].m_cKeys].m_pfRotation);

				rgAnims[iShape].m_cKeys++;
			}
		}
	}

	DtFrameSet((int)timeCurrent.value());

	delete[] rgrgiKeys;

	return hr;
}

*/







/*** TOOLS
 
	// print out the attributes of a dependency node
	for (unsigned iAttr = 0; iAttr < fnNode.attributeCount(); iAttr++) 
	{
		MFnAttribute fnAttr(fnNode.attribute(iAttr));

		cout << fnNode.name() << "." << fnAttr.name() << ": " << fnNode.attribute(iAttr).apiTypeStr() << endl;
	}





	// print the node containing the corresponging plug to 'plug'
	MPlugArray rgPlugs;

	plug.connectedTo(rgPlugs, true, true);

	for (int i = 0; i < (int)rgPlugs.length(); i++) 
	{
		MFnDependencyNode fnNode(rgPlugs[i].node());

		cout << fnNode.name() << "\t" << rgPlugs[i].name() << endl;
	}
 ***/